//
//
// Opis: Ilustruje uzyskiwanie informacji od systemu plikw proc
//
//
unit Procs;

interface

uses SysUtils, Libc, classes;

type
  // Procedura ktora otrzymuje  informacje o procesie od funkcji EnumProcesses
  TProcEnumCallback = procedure (pid: __pid_t) of object;

{
  EnumProcesses wywoluje dostarczana funkcje zwrotna dla kazdego
  procesu znalezionego w systemie plikow /proc.
}
procedure EnumProcesses (cb : TProcEnumCallback);

{
  Procedura ProcReadFile odczytuje zawartosc pliku dla
  wskazanego identyfikatora procesu; odczytana zawartosc
  zwraca we wskazanym buforze. Podczas wejscia do funkcji
  parametr length zawiera rozmiar bufora. Przy wyjsciu, parametr
  length zawiera ilosc odczytanych bajtow. W przypadku powodzenia
  funkcja zwraca wartosc 0, zas w przypadku bledu zwraca wartosc -1.
}
function  ProcReadFile (pid: __pid_t; const path: String;
  var buffer; var length: Integer): integer;

{
  ProcReadFileString returns the entire contents of the specified
  file in a string.
}
function ProcReadFileString (pid: __pid_t;
  const path: String): String;

// ProcGetCmdline - zwraca linie polecenia uzyta
// do uruchomienia procesu.
function  ProcGetCmdline (pid: __pid_t): String;

// ProcGetStatus - return the entire contents of the process'
// status file.
function  ProcGetStatus (pid: __pid_t): String;

// ProcParseStatusField - Parse a field's value from the
// passed status file contents.
function  ProcParseStatusField (const sStatus: String;
  const sField: String): String;

// ProcGetStatusField - Return the value of a single field
// in the status file.
function  ProcGetStatusField (pid: __pid_t; const
  sField: String): String;

// ProcGetName - zwraca nazwe procesu.
function  ProcGetName (pid: __pid_t): String;

// ProcReadLink - zwraca nazw pliku wskazywanego przez lacze
function  ProcReadLink (pid: __pid_t; const sFile: String): String;

type
  // Funkcja zwrotna dla wyliczania plikow otwartych przez proces.
  TProcFileEnumCallback = procedure (pid: __pid_t;
    fd: Integer) of object;

{
  ProcEnumFiles - wywoluje dostarczona funkcje zwrotna jednokrotnie
  dla kazdego pliku otwartego przez proces.
}
procedure ProcEnumFiles (pid: __pid_t; cb: TProcFileEnumCallback);

// ProcGetFD - zwraca zawartosc lacza dla pliku fd.
function  ProcGetFD (pid: __pid_t; fd: Integer): String;

// ProcGetCwd - zwraca biezaca kartoteke robocza procesu.
function  ProcGetCwd (pid: __pid_t): String;

// ProcGetExe - zwraca nazwe pliku wykonywalnego dla procesu.
function  ProcGetExe (pid: __pid_t): String;

// ProcGetRoot - zwraca glowna kartoteke procesu
function  ProcGetRoot (pid: __pid_t): String;

// ProcGetEnvironment - zwarca liste lancuchow zawierajaca
// lancuchy srodowiska procesu.
function  ProcGetEnvironment (pid: __pid_t): TStringList;

type
  // Rekord TProcStat zawiera informacje przechowywane
  // w pliku stat procesu.
  // Patrz "man 5 proc" w celu uzyskania informacji o tych polach.
  TProcStat = record
    pid         : __pid_t;  // identyfikator procesu
    comm        : String;   // nazwa pliku wykonywalnego
    state       : char;     // RSDZT
    ppid        : __pid_t;  // identyfikator procesu nadrzednego
    pgrp        : __pid_t;  // identyfikator grupy procesu
    session     : integer;  // identyfikator sesji
    tty         : integer;  // tty uzywany przez proces
    tpgid       : __pid_t;  // pid wlasciciela tty
    flags       : cardinal; //
    minflt      : cardinal; // ilosc mniej waznych wyjatkow dla procesu
    cminflt     : cardinal; // ilosc mniej waznych wyjatkow dla procesu
                            // oraz procesow potomnych
    majflt      : cardinal; // ilosc bardziej waznych wyjatkow dla procesu
    cmajflt     : cardinal; // ilosc bardziej waznych wyjatkow dla procesu
                            // oraz procesow potomnych
    utime       : cardinal; // czas spedzony w trybie uzytkownika
    stime       : cardinal; // czas spedzony w trybie jadra
    cutime      : cardinal; // czas spedzony w trybie uzytkownika
                            // wraz z procesami potomnymi
    cstime      : cardinal; // czas spedzony w trybie jadra
                            // wraz z procesami potomnymi
    counter     : integer;  // nastepny przydzial czasu
    priority    : integer;  // biezacy poziom priorytetu, zwiekszony o 15
    timeout     : cardinal; // czas nastepnego timeoutu
    itrealvalue : cardinal; // nastepny czas SIGALARM
    starttime   : integer;  // czas uruchomienia procesu
    vsize       : cardinal; // rozmiar pamieci wirtualnej
    rss         : cardinal; // rozmiar zestawu rezydentnego
    rlim        : cardinal; // limit RSS
    startcode   : cardinal; // poczatek przestrzeni kodu
    endcode     : cardinal; // koniec przestrzeni kodu
    startstack  : cardinal; // adres poczatku stosu
    kstkesp     : cardinal; // biezacy wskaznik stosu
    kstkeip     : cardinal; // biezacy wskaznik instrukcji
    signal      : integer;  // bitmapa oczekujacych sygnalow
    blocked     : integer;  // bitmapa zablokowanych sygnalow
    sigignore   : integer;  // bitmapa zignorowanych sygnalow
    sigcatch    : integer;  // bitmapa wychwyconych sygnalow
    wchan       : cardinal; // kanal
  end;

// ProcGetStat - Zwraca rekord TProcStat zawierajacy
// informacje z pliku stat procesu.
function ProcGetStat (pid: __pid_t) : TProcStat;

implementation

procedure EnumProcesses (cb : TProcEnumCallback);
var
  sr : TSearchRec;
  findRslt : Integer;
  pid : __pid_t;
begin
  findRslt := FindFirst ('/proc/*', faDirectory or faAnyFile, sr);
  try
    while findRslt = 0 do
    begin
      if ((sr.Attr and faDirectory) <> 0) then
      begin
        {
          Probujemy przekonwertowac nazwe kartoteki na liczbe calkowita.
          Jesli sie to nie powiedzie, wiemy ze nie jest to identyfikator
		  procesu.
        }
        try
          pid := StrToInt (sr.Name);
        except
          // to nie jest identyfikator procesu, wiec ustawiamy na -1
          pid := -1;
        end;
        if pid >= 0 then
          cb (pid);
      end;
      findRslt := FindNext (sr);
    end;
  finally
    FindClose (sr);
  end;
end;

function  ProcReadFile (pid: __pid_t; const path: String;
  var buffer; var length: Integer): integer;
var
  sFileName : String;
  fd : Integer;
begin
  Result := -1;
  try
    sFileName := Format ('/proc/%d/%s', [pid, path]);

    fd := Libc.open (PChar (sFilename), O_RDONLY);
    if fd <> -1 then
    begin
      try
        Result := Libc.__read (fd, Buffer, length);
      finally
        Libc.__close(fd);
      end;
    end;
  except
    // pomijamy wyjatek...
  end;
end;

function ProcReadFileString (pid: __pid_t;
  const path: String): String;
var
  buffer : array [0..16000] of char;
  length : Integer;
begin
  length := sizeof (buffer)-1;
  length := ProcReadFile (pid, path, buffer, length);
  if length = -1 then
    Result := ''
  else
  begin
    buffer[length] := #0;
    Result := buffer;
  end;
end;

function ProcGetCmdline (pid: __pid_t) : String;
begin
  Result := ProcReadFileString (pid, 'cmdline');
end;

function ProcGetStatus (pid: __pid_t): String;
begin
  Result := ProcReadFileString (pid, 'status');
end;

{
 ProcParseStatusField - Returns the value of the specified
 field by parsing it from sStatus.
}
function ProcParseStatusField (const sStatus: String;
  const sField: String): String;
var
  pStatus : PChar;
  pField,
  pEnd : PChar;
  iStart : Integer;
  iLength : Integer;
begin
  Result := '';
  pStatus := PChar(sStatus+':');
  // locate the field
  pField := StrPos (pStatus, PChar(sField));
  if pField <> nil then
  begin
    // skip beyond the field and the trailing colon
    pField := pField + Length(sField) + 1;
    // locate the end of line
    pEnd := StrPos (pField, #10);
    if pEnd = nil then
      pEnd := StrPos (pField, #0);
    dec (pEnd);
    // and skip over white space
    while (pField^ in [#9,' ']) and (pField^ <> #0) do
      inc (pField);
    // copy everything after the field name to end of line
    iStart := pField-pStatus+1;
    iLength := pEnd-pField+1;
    Result := system.Copy (sStatus, iStart, iLength);
  end;
end;

function ProcGetStatusField (pid: __pid_t;
  const sField: String): String;
begin
  Result := ProcParseStatusField (ProcGetStatus (pid), sField);
end;

function ProcGetName (pid: __pid_t): String;
begin
  Result := ProcGetStatusField (pid, 'Name');
end;

function  ProcReadLink (pid: __pid_t; const sFile: String): String;
var
  buf : array [0..1024] of char;
  iLen : Integer;
  sFilename : String;
begin
  sFilename := Format ('/proc/%d/%s', [pid, sFile]);
  iLen := readlink (PChar(sFilename), buf, 1024);
  if iLen = -1 then
    iLen := 0;
  buf[iLen] := #0;
  Result := buf;
end;

function  ProcGetFD (pid: __pid_t; fd: Integer): String;
begin
  Result := ProcReadLink (pid, Format ('fd/%d', [fd]));
end;

{
  ProcEnumFiles - Wylicza pliki uzywane przez proces.
  Jesli wywolujacy proces nie ma uprawnien dla dostepu
  do kartoteki fd procesu, nie zostana wyliczone zadne
  deskryptory plikow. Nie spowoduje to bledu.
}
procedure ProcEnumFiles (pid: __pid_t; cb: TProcFileEnumCallback);
var
  sr : TSearchRec;
  findRslt : Integer;
  sFilename : String;
  fd : Integer;
begin
  sFilename := Format ('/proc/%d/fd/*', [pid]);
  findRslt := FindFirst (sFilename, faAnyFile, sr);
  try
    while findRslt = 0 do
    begin
      try
        fd := StrToInt (sr.Name);
      except
        fd := -1;
      end;
      if fd <> -1 then
        cb (pid, fd);
      findRslt := FindNext (sr);
    end;
  finally
    FindClose (sr);
  end;
end;

function  ProcGetCwd (pid: __pid_t): String;
begin
  Result := ProcReadLink (pid, 'cwd');
end;

function  ProcGetExe (pid: __pid_t): String;
begin
  Result := ProcReadLink (pid, 'exe');
end;

function  ProcGetRoot (pid: __pid_t): String;
begin
  Result := ProcReadLink (pid, 'root');
end;

{
  ProcGetStat - Zwraca szczegolowe informacje o procesie
  w rekordzie TProcStat
}
function ProcGetStat (pid: __pid_t) : TProcStat;
const
  // lancuch scanf dla pliku stat
  StatScanfFormat =
    '%d %s %c %d %d %d %d %d %u %u %u %u %u %u %u %u %u %d %d %u '+
    '%u %d %u %u %u %u %u %u %u %u %d %d %d %d %u';
var
  s : TProcStat;
  pcomm : PChar;
  stat : String;
begin
  stat := ProcReadFileString (pid, 'stat');
  // alokuje lancuch dla nazwy pliku wykonywalnego.
  // jest on dluzszy niz to absolutnie koniecznie, lecz to nie szkodzi
  pcomm := StrAlloc (Length (stat));
  try
    sscanf (PChar (stat), StatScanfFormat,
      @s.pid, pcomm,   @s.state,        @s.ppid,
      @s.pgrp,         @s.session,      @s.tty,
      @s.tpgid,        @s.flags,        @s.minflt,
      @s.cminflt,      @s.majflt,       @s.cmajflt,
      @s.utime,        @s.stime,        @s.cutime,
      @s.cstime,       @s.counter,      @s.priority,
      @s.timeout,      @s.itrealvalue,  @s.starttime,
      @s.vsize,        @s.rss,          @s.rlim,
      @s.startcode,    @s.endcode,      @s.startstack,
      @s.kstkesp,      @s.kstkeip,      @s.signal,
      @s.blocked,      @s.sigignore,    @s.sigcatch,
      @s.wchan);
    s.comm := pcomm;
  finally
    StrDispose (pcomm);
  end;
  Result := s;
end;

function  ProcGetEnvironment (pid: __pid_t): TStringList;
var
  buffer : array[0..16000] of char;
  length : integer;
  lst : TStringList;
  p : PChar;
begin
  length := sizeof (buffer)-1;
  length := ProcReadFile (pid, 'environ', buffer, length);
  if length = -1 then
  begin
    Result := nil;
    exit;
  end;
  buffer[length] := #0;

  // zwrocone srodowisko jest seria zakonczonych zerem lancuchow
  lst := TStringList.Create;
  try
    p := buffer;
    while p^ <> #0 do
    begin
      lst.Add (p);
      p := StrEnd (p) + 1;
    end;
  except
    lst.Free;
    raise;
  end;
  Result := lst;
end;

end.

