package rozdzial8;

import java.io.*;
import java.math.BigInteger;
import java.security.cert.*;
import java.util.*;
import javax.security.auth.x500.X500PrivateCredential;

import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.jce.PKCS10CertificationRequest;
import org.bouncycastle.openssl.PEMReader;
import org.bouncycastle.x509.X509V3CertificateGenerator;
import org.bouncycastle.x509.extension.*;

import rozdzial9.Utils;

/**
 * Przykad ilustrujcy przetwarzanie dania certyfikacji PKCS #10, zapisanego w pliku
 * "pkcs10.req" w formacie PEM. W ramach odpowiedzi generowana jest cieka certyfikatw
 * PKCS #7, umieszczana w pliku "pkcs7.pth". 
 * <p>
 * Wygenerowany certyfikat i jego cieka bd wane przez 50 sekund.
 */
public class CertReqSolution
{
    public static void main(String[] args)
        throws Exception
    {
        // utworzenie certyfikatw centrum certyfikacji
        X500PrivateCredential rootCredential = Utils.createRootCredential();
        X500PrivateCredential interCredential = Utils.createIntermediateCredential(
                rootCredential.getPrivateKey(), rootCredential.getCertificate());
        
        // przetworzenie dania
        PEMReader   pRd = new PEMReader(
                                 new InputStreamReader(
                                         new FileInputStream("pkcs10.req")));
        
        PKCS10CertificationRequest request = (PKCS10CertificationRequest)pRd.readObject();

        // pobranie certyfikatu walidujcego
        X509Certificate caCert = interCredential.getCertificate();
        
        X509V3CertificateGenerator  certGen = new X509V3CertificateGenerator();

        certGen.setSerialNumber(BigInteger.valueOf(System.currentTimeMillis()));
        certGen.setIssuerDN(caCert.getSubjectX500Principal());
        certGen.setNotBefore(new Date(System.currentTimeMillis()));
        certGen.setNotAfter(new Date(System.currentTimeMillis() + 50000));
        certGen.setSubjectDN(request.getCertificationRequestInfo().getSubject());
        certGen.setPublicKey(request.getPublicKey("BC"));
        certGen.setSignatureAlgorithm("SHA256WithRSAEncryption");
        
        // dostarczenie kilku podstawowych rozszerze i oznaczenie certyfikatu jako uprawnionego do podpisywania i szyfrowania
        certGen.addExtension(X509Extensions.AuthorityKeyIdentifier, false, new AuthorityKeyIdentifierStructure(caCert));
        
        certGen.addExtension(X509Extensions.SubjectKeyIdentifier, false, new SubjectKeyIdentifierStructure(request.getPublicKey("BC")));
        
        certGen.addExtension(X509Extensions.BasicConstraints, true, new BasicConstraints(false));
        
        certGen.addExtension(X509Extensions.KeyUsage, true, new KeyUsage(KeyUsage.digitalSignature | KeyUsage.keyEncipherment));
        
        // utworzenie cieki
        List chain = Arrays.asList(
                new Certificate[] { 
                        certGen.generateX509Certificate(interCredential.getPrivateKey(), "BC"), 
                        interCredential.getCertificate(), 
                        rootCredential.getCertificate() });
        
        // utworzenie obiektu CertPath dla cieki
        CertificateFactory fact = CertificateFactory.getInstance("X.509", "BC");
        
        CertPath path = fact.generateCertPath(chain);
        
        // zapisanie wyniku do pliku
        FileOutputStream fOut = new FileOutputStream("pkcs7.pth");
        
        fOut.write(path.getEncoded("PKCS7"));
        
        fOut.close();
    }
}
