package rozdzial5;

import java.util.Date;

import org.bouncycastle.asn1.*;

/**
 * Implementacja przykadowej struktury ASN.1.
 * <pre>
 * MyStructure ::= SEQUENCE {
 *                   version INTEGER DEFAULT 0,
 *                   created GeneralizedTime,
 *                   baseData OCTET STRING,
 *                   extraData [0] UTF8String OPTIONAL,
 *                   commentData [1] UTF8String OPTIONAL }
 * <pre>
 * 
 */
public class MyStructure extends ASN1Encodable
{
    private DERInteger         version;
    private DERGeneralizedTime created;
    private ASN1OctetString    baseData;
    private DERUTF8String      extraData = null;
    private DERUTF8String      commentData = null;
    
    /**
     * Konstruktor pobierajcy obiekt ASN.1 SEQUENCE.
     */
    public MyStructure(
        ASN1Sequence    seq)
    {
        int index = 0;
        
        // sprawdzenie pola version
        if (seq.getObjectAt(0) instanceof DERInteger)
        {
            this.version = (DERInteger)seq.getObjectAt(0);
            index++;
        }
        else
        {
            this.version = new DERInteger(0);
        }
        
        this.created = (DERGeneralizedTime)seq.getObjectAt(index++);
        this.baseData = (ASN1OctetString)seq.getObjectAt(index++);
        
        // sprawdzenie pl nieobowizkowych
        for (int i = index; i != seq.size(); i++)
        {
            ASN1TaggedObject  t = (ASN1TaggedObject)seq.getObjectAt(i);
            
            switch (t.getTagNo())
            {
            case 0:
                extraData = DERUTF8String.getInstance(t, false);
                break;
            case 1:
                commentData = DERUTF8String.getInstance(t, false);
                break;
            default:
                throw new IllegalArgumentException("Nieznany znacznik " + t.getTagNo() + " w konstruktorze");
            }
        }
    }
    
    /**
     * Konstruktor pobierajcy zestaw obiektw i typw pierwotnych Javy 
     * odpowiadajcych poszczeglnym polom struktury.
     */
    public MyStructure(
        int     version,
        Date    created,
        byte[]  baseData,
        String  extraData,
        String  commentData)
    {
        this.version = new DERInteger(version);
        this.created = new DERGeneralizedTime(created);
        this.baseData = new DEROctetString(baseData);
        
        if (extraData != null)
        {
            this.extraData = new DERUTF8String(extraData);
        }
        
        if (commentData != null)
        {
            this.commentData = new DERUTF8String(commentData);
        }
    }
    
    /* 
     * Generuje obiekt nadajcy si do zapisu do strumienia ASN1/DEROutputStream.
     */
    public DERObject toASN1Object()
    {
        ASN1EncodableVector   v = new ASN1EncodableVector();
        
        if (version.getValue().intValue() != 0)
        {
            v.add(version);
        }
        
        v.add(created);
        v.add(baseData);
        
        if (extraData != null)
        {
            v.add(new DERTaggedObject(false, 0, extraData));
        }
        
        if (commentData != null)
        {
            v.add(new DERTaggedObject(false, 1, commentData));
        }
        
        return new DERSequence(v);
    }
}
