package com.skatestown.xml;

import java.text.NumberFormat;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import javax.xml.parsers.DocumentBuilder;
import com.skatestown.data.PO;
import com.skatestown.data.POItem;
import com.skatestown.data.Invoice;
import com.skatestown.data.InvoiceItem;
import com.skatestown.data.Address;
import com.skatestown.data.BusinessDocument;

public class Serializer {
	static final String PO_NS = "http://www.skatestown.com/ns/po";
	static final String INVOICE_NS = "http://www.skatestown.com/ns/invoice";
	static final String XSI_NS = "http://www.w3.org/2001/XMLSchema-instance";
	static final String PO_XSD = "http://www.skatestown.com/schema/invoice.xsd";
	static final String INVOICE_XSD = "http://www.skatestown.com/schema/invoice.xsd";
	
	Document doc;
	NumberFormat nf = NumberFormat.getInstance();
	
	public static Document writePO(DocumentBuilder builder, PO po) {
		Document doc = builder.newDocument();
		
		Element root = doc.createElement("po");
		root.setAttribute("xmlns", PO_NS);
		root.setAttribute("xmlns:xsi", XSI_NS);
		root.setAttribute("xsi:schemaLocation", PO_XSD);
		
		doc.appendChild(root);
		
		Serializer ser = new Serializer(doc);
		ser.writePO(po, root);
		
		return doc;
	}
	
	public static Document writeInvoice(DocumentBuilder builder,
                                        Invoice invoice)
    {
		Document doc = builder.newDocument();
		
		Element root = doc.createElement("invoice");
		root.setAttribute("xmlns", INVOICE_NS);
		root.setAttribute("xmlns:xsi", XSI_NS);
		root.setAttribute("xsi:schemaLocation", INVOICE_XSD);
		
		doc.appendChild(root);
		
		Serializer ser = new Serializer(doc);
		ser.writeInvoice(invoice, root);
		
		return doc;
	}
	
	protected Serializer(Document doc) {
		this.doc = doc;
		nf.setMinimumFractionDigits(2);
		nf.setMaximumFractionDigits(2);
		nf.setGroupingUsed(false);
	}
	
	protected Element addChild(Element elem, String name) {
		Element newElem = doc.createElement(name);
		elem.appendChild(newElem);
		return newElem;
	}
	
	protected Element addChild(Element elem, String name, String value) {
        if (value == null)
            return null;
        Element newElem = doc.createElement(name);
        newElem.appendChild(doc.createTextNode(value));
        elem.appendChild(newElem);
        return newElem;
	}
	
	protected void addChildren(Element elem, String name, String[] values) {
		for (int i = 0; i < values.length; i++) {
			Element newElem = doc.createElement(name);
			newElem.appendChild(doc.createTextNode(values[i]));
			elem.appendChild(newElem);
		}
	}
	
	protected void writeDocument(BusinessDocument bdoc, Element elem) {
		elem.setAttribute("id", ""+bdoc.getId());
		elem.setAttribute("submitted", bdoc.getDate());
		writeAddress(bdoc.getBillTo(), addChild(elem, "billTo"));
		writeAddress(bdoc.getShipTo(), addChild(elem, "shipTo"));
	}
	
	protected void writeAddress(Address addr, Element elem) {
		addChild(elem, "name", addr.getName());
		addChild(elem, "company", addr.getCompany());
		addChildren(elem, "street", addr.getStreet());
		addChild(elem, "city", addr.getCity());
		addChild(elem, "state", addr.getState());
		addChild(elem, "postalCode", addr.getPostalCode());
		addChild(elem, "country", addr.getCountry());
	}
	
	protected void writePOItem(POItem item, Element elem) {
		elem.setAttribute("sku", item.getSKU());
		elem.setAttribute("quantity", ""+item.getQuantity());
		addChild(elem, "description", item.getDescription());
	}
	
	protected void writeInvoiceItem(InvoiceItem item, Element elem) {
		writePOItem(item, elem);
		elem.setAttribute("unitPrice", nf.format(item.getUnitPrice()));
	}
	
	protected void writePO(PO po, Element elem) {
		writeDocument(po, elem);
		Element order = addChild(elem, "order");
		POItem[] items = po.getItems();
		for (int i = 0; i < items.length; i++)
			writePOItem(items[i], addChild(order, "item"));
	}
	
	protected void writeInvoice(Invoice invoice, Element elem) {
		writeDocument(invoice, elem);
		Element order = addChild(elem, "order");
		InvoiceItem[] items = invoice.getItems();
		for (int i = 0; i < items.length; i++)
			writeInvoiceItem(items[i], addChild(order, "item"));
		
		addChild(elem, "tax", nf.format(invoice.getTax()));
		addChild(elem,
                 "shippingAndHandling",
                 nf.format(invoice.getShippingAndHandling()));
		addChild(elem, "totalCost", nf.format(invoice.getTotalCost()));
	}
}
