package com.skatestown.invoice;

import java.io.InputStream;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Check SkatesTown invoice totals using a SAX parser.
 */
public class InvoiceCheckerSAX
    extends DefaultHandler
    implements InvoiceChecker
{
    // invoice running total
    double runningTotal = 0.0;

    // invoice total
    double total = 0.0;
    
    // Utility data for extracting money amounts from content
    boolean isMoneyContent = false;
    double amount = 0.0;
    
    /**
     * Check invoice totals.
     * @param invoiceXML Invoice XML document
     * @exception Exception Any exception returned during checking
     */
    public void checkInvoice(InputStream invoiceXML) throws Exception {
        // Use the default (non-validating) parser
        SAXParserFactory factory = SAXParserFactory.newInstance();
        SAXParser saxParser = factory.newSAXParser();

        // Parse the input; we are the handler of SAX events
        saxParser.parse(invoiceXML, this);
    }

    // SAX DocumentHandler methods
    public void startDocument() throws SAXException {
        runningTotal = 0.0;
        total = 0.0;
        isMoneyContent = false;
    }

    public void endDocument() throws SAXException {
        // Use delta equality check to prevent cumulative
        // binary arithmetic errors. In this case, the delta
        // is one half of one cent
        if (Math.abs(runningTotal - total) >= 0.005) {
            throw new SAXException(
                "Invoice error: total is " + Double.toString(total) +
                " while our calculation shows a total of " +
                Double.toString(Math.round(runningTotal * 100) / 100.0));
        }
    }

    public void startElement(String namespaceURI,
                             String localName,
                             String qualifiedName,
                             Attributes attrs) throws SAXException {
        if (localName.equals("item")) {
            // Find item subtotal; add it to running total
            runningTotal +=
                Integer.valueOf(attrs.getValue(namespaceURI,
                    "quantity")).intValue() *
                Double.valueOf(attrs.getValue(namespaceURI,
                    "unitPrice")).doubleValue();
        } else if (localName.equals("tax") ||
                   localName.equals("shippingAndHandling") ||
                   localName.equals("totalCost")) {
            // Prepare to extract money amount
            isMoneyContent = true;
        }
    }

    public void endElement(String namespaceURI,
                           String localName,
                           String qualifiedName) throws SAXException {
        if (isMoneyContent) {
            if (localName.equals("totalCost")) {
                total = amount;
            } else {
                // It must be tax or shippingAndHandling
                runningTotal += amount;
            }
            isMoneyContent = false;
        }
    }

    public void characters(char buf[], int offset, int len)
        throws SAXException {
        if (isMoneyContent) {
            String value = new String(buf, offset, len);
            amount = Double.valueOf(value).doubleValue();
        }
    }
}
