/*
 * @(#) $Id: Account.java,v 1.1.1.1 2003/10/05 18:39:18 pankaj_kumar Exp $
 *
 * Copyright (c) 2002-03 by Pankaj Kumar (http://www.pankaj-k.net). 
 * All rights reserved.
 *
 * The license governing the use of this file can be found in the 
 * root directory of the containing software.
 */
package org.jstk.example.bank.server;

import java.math.BigDecimal;
import java.util.Vector;
import java.util.Date;
import java.security.AccessController;
import org.jstk.example.bank.*;

public class Account implements AccountIntf, java.io.Serializable {
	private static BigDecimal ZERO = new BigDecimal("0.00");
	private String acctNo;
	private boolean closed;
	private BigDecimal balance = null;
	private Vector transactions = null;
	private transient BankPersistenceManagerIntf pmi = null;

	public Account(String acctNo){
		this.acctNo = acctNo;
		checkPermission("open");
		Transaction trn = new Transaction(Transaction.OPEN, ZERO, ZERO, "account open");
		transactions = new Vector();
		transactions.add(trn);
		balance = ZERO;
		closed = false;
	}

	public void setPersistenceManager(BankPersistenceManagerIntf pmi){
		this.pmi = pmi;
	}

	public void deposit(BigDecimal amt) throws Exceptions.AccountClosed{
		checkPermission("deposit");
		if (closed)
			throw new Exceptions.AccountClosed();
		balance = balance.add(amt);
		Transaction trn = new Transaction(Transaction.CREDIT, amt, balance, "cash deposit");
		transactions.add(trn);
		if (pmi != null)
			pmi.save();
	}
	public void withdraw(BigDecimal amt) throws Exceptions.AccountClosed, Exceptions.InsufficientAmount {
		checkPermission("withdraw");
		if (closed)
			throw new Exceptions.AccountClosed();
		if (balance.compareTo(amt) <= 0)
			throw new Exceptions.InsufficientAmount();
		balance = balance.subtract(amt);
		Transaction trn = new Transaction(Transaction.DEBIT, amt, balance, "cash withdrawal");
		transactions.add(trn);
		if (pmi != null)
			pmi.save();
	}

	public void close() throws Exceptions.AccountClosed {
		checkPermission("close");
		if (closed)
			throw new Exceptions.AccountClosed();
		BigDecimal amt = new BigDecimal("0.00");
		Transaction trn = new Transaction(Transaction.CLOSE, amt, balance, "account close");
		transactions.add(trn);
		closed = true;
	}
	public BigDecimal getBalance() throws Exceptions.AccountClosed{
		checkPermission("read");
		if (closed)
			throw new Exceptions.AccountClosed();
		return balance;
	}
	public String getAcctNo(){
		checkPermission("read");
		return acctNo;
	}
	public String getStatement(){
		checkPermission("read");
		String status = (closed ? "CLOSED" : "OPEN");
		StringBuffer sb = new StringBuffer();
		sb.append("----------------- BEGIN BANK STATEMENT -----------------\n");
		sb.append("Statement Date : " + new Date() + "\n");
		sb.append("Account#       : " + acctNo + "\n");
		sb.append("Account Status : " + status + "\n");
		sb.append("Transactions   :\n");
		for (int i = 0; i < transactions.size(); i++){
			sb.append(transactions.elementAt(i) + "\n");
		}
		sb.append("------------------ END BANK STATEMENT ------------------\n");
		return sb.toString();
	}
	private void checkPermission(String action){
		SecurityManager security = System.getSecurityManager();
		if (security != null) {
			AccountPermission ap = new AccountPermission(acctNo, action);
			AccessController.checkPermission(ap);
		}
	}

	public String toString(){
		String status = (closed ? "CLOSED" : "OPEN");
		return "Account#: " + acctNo + ", Balance: " + balance + ", Status: " + status;
	}
}