package coreservlets;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.util.*;
import java.text.NumberFormat;

/** Wywietla informacje o towarach, ktre aktualnie
 *  znajduj si w koszyku (ShoppingCart). Przegldarki
 *  maj swoje wasne sesje, na podstawie ktrych 
 *  okrelane jest przynaleno koszykw. Jeli to 
 *  jest pierwsza wizyta na stronie umoliwiajcej 
 *  skadnie zamwie, to tworzony jest nowy koszyk.
 *  Uytkownicy zazwyczaj trafiaj na t stron ze stron
 *  prezentujcych towary, ktre mona zamawia, dlatego
 *  te ta strona dodaje nowy element do koszyka. Jednak
 *  uytkownicy mog zapamita adres tej strony i wywietla
 *  j posugujc si list ulubionych stron; mog take
 *  wrci na ni klikajc przycisk "Aktualizuj zamwienie"
 *  po zmienieniu iloci egzemplarzy jednego z zamawianych 
 *  towarw.
 *  <P>
 *  Przykady z ksiki Java Servlet i JavaServer Pages
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2000 Marty Hall; mona kopiowa i modyfikowa bez ogranicze.
 */

public class OrderPage extends HttpServlet {
  public void doGet(HttpServletRequest request,
                    HttpServletResponse response)
      throws ServletException, IOException {
    HttpSession session = request.getSession(true);
    ShoppingCart cart;
    synchronized(session) {
      cart = (ShoppingCart)session.getValue("shoppingCart");
      // Dla nowych uytkownikw tworzone s nowe koszyki.
      // Uytkownicy, ktrzy ju dysponuj koszykami 
      // uywaj tych, ktre zostay dla nich wczeniej utworzone .
      if (cart == null) {
        cart = new ShoppingCart();
        session.putValue("shoppingCart", cart);
      }
      String itemID = request.getParameter("itemID");
      if (itemID != null) {
        String numItemsString =
          request.getParameter("numItems");
        if (numItemsString == null) {
          // Jeli w daniu zosta podany identyfikator (ID) lecz
          // nie liczba, to oznacza to, e uytkownik trafi
          // tutaj klikajc przycisk "Dodaj do koszyka" na jednej
          // ze stron prezentujcej towary z katalogu.
          cart.addItem(itemID);
        } else {
          // Jeli w daniu zosta podany zarwno identyfikator 
          // (ID) jak i liczba, to oznacza to, e uytkownik 
          // trafi na stron klikajc przycisk "Aktualizuj
          // zamwienie" po zmianie iloci egzemplarzy jednego
          // z zamawianych towarw. Zwr uwag, i podanie wartoci
          // 0 jako liczby egzemplarzy zamawianego towaru sprawi, e 
          // dany towar zostanie usunity z koszyka.
          int numItems;
          try {
            numItems = Integer.parseInt(numItemsString);
          } catch(NumberFormatException nfe) {
            numItems = 1;
          }
          cart.setNumOrdered(itemID, numItems);
        }
      }
    }
    // Poka status zamwienia niezalenie od tego czy uytkownik 
    // je zmodyfikowa czy nie.
    response.setContentType("text/html; charset=ISO-8859-2");
    PrintWriter out = response.getWriter();
    String title = "Status zamwienia";
    out.println(ServletUtilities.headWithTitle(title) +
                "<BODY BGCOLOR=\"#FDF5E6\">\n" +
                "<H1 ALIGN=\"CENTER\">" + title + "</H1>");
    synchronized(session) {
      Vector itemsOrdered = cart.getItemsOrdered();
      if (itemsOrdered.size() == 0) {
        out.println("<H2><I>Brak towarw w koszyku...</I></H2>");
      } else {
        // Jeli w koszyku jest co najmniej jeden towar, to
        // wywietl tabel z informacjami o nim.
        out.println
          ("<TABLE BORDER=1 ALIGN=\"CENTER\">\n" +
           "<TR BGCOLOR=\"#FFAD00\">\n" +
           "  <TH>Identyfikator<TH>Opis\n" +
           "  <TH>Cena jednostkowa<TH>Ilo<TH>Warto");
        ItemOrder order;
        
        // Zaokrglamy do dwch miejsc po przecinku, 
        // wstawiamy znak dolara (lub innej waluty), itd.
        // wszystko zgodnie z biecymi ustawieniami lokalnymi.
        NumberFormat formatter =
          NumberFormat.getCurrencyInstance();

        String formURL =
          "/servlet/coreservlets.OrderPage";
        // Adresy URL odwoujce si do stron tej samej witryny
        // przeksztacamy przy uyciu metody encodeURL.
        formURL = response.encodeURL(formURL);
        
        // Dla kadego towaru umieszczonego w koszuku
        // tworzymy wiersz tabeli zawierajcy identyfikato
        // towaru (ID), opis, jego cen jednostkow,
        // ilo zamwionych egzemplarzy oraz czn cen.
        // Ilo zamawianych egzemplarzy wywietlamy w 
        // polu tekstowym, tak aby uytkownik mg j zmieni.
        // Dodatkowo, obok pola, wywietlamy przycisk 
        // "Aktualizuj zamwienie", ktry powoduje ponowne 
        // przesanie tej samej strony na serwer, przy czym
        // zmieniana jest ilo zamawianych egzemplarzy 
        // danego towaru.
        for(int i=0; i<itemsOrdered.size(); i++) {
          order = (ItemOrder)itemsOrdered.elementAt(i);
          out.println
            ("<TR>\n" +
             "  <TD>" + order.getItemID() + "\n" +
             "  <TD>" + order.getShortDescription() + "\n" +
             "  <TD>" +
             formatter.format(order.getUnitCost()) + "\n" +
             "  <TD>" +
             "<FORM ACTION=\"" + formURL + "\">\n" +
             "<INPUT TYPE=\"HIDDEN\" NAME=\"itemID\"\n" +
             "       VALUE=\"" + order.getItemID() + "\">\n" +
             "<INPUT TYPE=\"TEXT\" NAME=\"numItems\"\n" +
             "       SIZE=3 VALUE=\"" + 
             order.getNumItems() + "\">\n" +
             "<SMALL>\n" +
             "<INPUT TYPE=\"SUBMIT\"\n "+
             "       VALUE=\"Aktualizuj zamwienie\">\n" +
             "</SMALL>\n" +
             "</FORM>\n" +
             "  <TD>" +
             formatter.format(order.getTotalCost()));
        }
        String checkoutURL =
          response.encodeURL("/Checkout.html");
        // Pod tabel wywietlany jest przycisk "Rozliczenie"
        out.println
          ("</TABLE>\n" +
           "<FORM ACTION=\"" + checkoutURL + "\">\n" +
           "<BIG><CENTER>\n" +
           "<INPUT TYPE=\"SUBMIT\"\n" +
           "       VALUE=\"Rozliczenie\">\n" +
           "</CENTER></BIG></FORM>");
      }
      out.println("</BODY></HTML>");
    }
  }

  /** dania GET i POST s obsugiwane w identyczny sposb */
  
  public void doPost(HttpServletRequest request,
                     HttpServletResponse response)
      throws ServletException, IOException {
    doGet(request, response);
  }
}
