package coreservlets;

import java.sql.*;

/** Trzy narzdzia uatwiajce korzystanie z baz danych:<BR>
 *   1) getQueryResults. Nawizuje poczenie z baz danych, wykonuje
 *      zapytanie, pobiera wszystkie wiersze wynikw w postaci tablic
 *      acuchw znakw i zapisuje je w obiekcie klasy DBResults.
 *      W obiekcie DBResults zapisywana jest take nazwa oprogramowania 
 *      serwera bazy danych, numer wersji serwera oraz nazwy wszystkich
 *      kolumn. Istniej dwie wersje tej metody:
 *      pierwsza, ktra tworzy nowe poczenie oraz druga, ktra
 *      wykorzystuje ju istniejce poczenie. <P>
 *   2) createTable. Na podstawie nazwy tabeli, acucha znakw okrelajcego 
 *      formaty kolumn oraz tablicy acuchw znakw podajcej wartoci 
 *      wierszy, metoda ta nawizuje poczenie z baz danych,
 *      usuwa aktualn wersj wskazanej tabeli 
 *      wykonuje polecenie SQL: CREATE TABLE wykorzystujc przy tym podane 
 *      formaty kolumn, a nastpnie generuje seri polece SQL: INSERT INTO
 *      powodujcych zapisanie informacji w nowej tabeli. Take ta metoda jest
 *      dostpna w dwch wersjach: pierwszej, ktra tworzy nowe poczenie
 *      oraz drugiej, ktra uywa poczenia ju istniejcego. <P>
 *   3) printTable. Na podstawie nazwy tabeli, metoda nawizuje poczenie 
 *      z okrelon baz danych, pobiera jej wszystkie wiersze 
 *      i wywietla je uywajc standardowego strumienia wyjciowego.
 *  <P>
 *  Przykady z ksiki Java Servlet i JavaServer Pages
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2000 Marty Hall; mona kopiowa i modyfikowa bez ogranicze.
 */

public class DatabaseUtilities {
  
  /** Nawizuje poczenie z baz danych, wykonuje podane
   *  zapytanie i zapisuje uzyskane wyniki w obiekcie DBRresults.
   *  Jeli poczenie z baz danych zostanie otworzone (okrela
   *  to argument "close"), to bdzie mona je pobra przy uyciu
   *  metody DBResults.getConnection.
   */
  
  public static DBResults getQueryResults(String driver,
                                          String url,
                                          String username,
                                          String password,
                                          String query,
                                          boolean close) {
    try {
      Class.forName(driver);
      Connection connection =
        DriverManager.getConnection(url, username, password);
      return(getQueryResults(connection, query, close));
    } catch(ClassNotFoundException cnfe) {
      System.err.println("Bd adowania sterownika: " + cnfe);
      return(null);
    } catch(SQLException sqle) {
      System.err.println("Bd przy nawizywaniu poczenia: " + sqle);
      return(null);
    }
  }

  /** Pobiera wyniki podobnie jak w poprzedniej metodzie, jednak 
   *  nie tworzy nowego poczenia a wykorzystuje poczenie ju 
   *  istniejce.
   */
  
  public static DBResults getQueryResults(Connection connection,
                                          String query,
                                          boolean close) {
    try {
      DatabaseMetaData dbMetaData = connection.getMetaData();
      String productName =
        dbMetaData.getDatabaseProductName();
      String productVersion =
        dbMetaData.getDatabaseProductVersion();
      Statement statement = connection.createStatement();
      ResultSet resultSet = statement.executeQuery(query);
      ResultSetMetaData resultsMetaData =
        resultSet.getMetaData();
      int columnCount = resultsMetaData.getColumnCount();
      String[] columnNames = new String[columnCount];
      // Indeksy kolumn rozpoczynaj si od 1 (jak w SQL-u) 
      // a nie od 0 (jak w jzyku Java).
      for(int i=1; i<columnCount+1; i++) {
        columnNames[i-1] =
          resultsMetaData.getColumnName(i).trim();
      }
      DBResults dbResults =
        new DBResults(connection, productName, productVersion,
                      columnCount, columnNames);      
      while(resultSet.next()) {
        String[] row = new String[columnCount];
        // Indeksy z obiekcie ResultSet rozpoczynaj si od 0.
        for(int i=1; i<columnCount+1; i++) {
          String entry = resultSet.getString(i);
          if (entry != null) {
            entry = entry.trim();
          }
          row[i-1] = entry;
        }
        dbResults.addRow(row);
      }
      if (close) {
        connection.close();
      }
      return(dbResults);
    } catch(SQLException sqle) {
      System.err.println("Bd przy nawizywaniu poczenia: " + sqle);
      return(null);
    } 
  }

  /** Tworzy tabel o okrelonym formacie i zapisuje w niej 
   *  podane wiersze danych
   */
  
  public static Connection createTable(String driver,
                                       String url,
                                       String username,
                                       String password,
                                       String tableName,
                                       String tableFormat,
                                       String[] tableRows,
                                       boolean close) {
    try {
      Class.forName(driver);
      Connection connection =
        DriverManager.getConnection(url, username, password);
      return(createTable(connection, username, password,
                         tableName, tableFormat,
                         tableRows, close));
    } catch(ClassNotFoundException cnfe) {
      System.err.println("Bd adowania sterownika: " + cnfe);
      return(null);
    } catch(SQLException sqle) {
      System.err.println("Bd przy nawizywaniu poczenia: " + sqle);
      return(null);
    } 
  }

  /** Podobna do poprzedniej metody, lecz uywa istniejcego poczenia. */
  
  public static Connection createTable(Connection connection,
                                       String username,
                                       String password,
                                       String tableName,
                                       String tableFormat,
                                       String[] tableRows,
                                       boolean close) {
    try {
      
      Statement statement = connection.createStatement();
      // Usuwa aktualn tabel jeli taka istnieje, lecz nie zgasza bdw
      // jeli tablie nie ma. Do tego celu suy osobny blok try/catch.
      try {
        statement.execute("DROP TABLE " + tableName);
      } catch(SQLException sqle) {}
      String createCommand =
        "CREATE TABLE " + tableName + " " + tableFormat;
      statement.execute(createCommand);
      String insertPrefix =
        "INSERT INTO " + tableName + " VALUES";
      for(int i=0; i<tableRows.length; i++) {
        statement.execute(insertPrefix + tableRows[i]);
      }
      if (close) {
        connection.close();
        return(null);
      } else {
        return(connection);
      }
    } catch(SQLException sqle) {
      System.err.println("Bd przy tworzeniu tabeli: " + sqle);
      return(null);
    } 
  }

  public static void printTable(String driver,
                                String url,
                                String username,
                                String password,
                                String tableName,
                                int entryWidth,
                                boolean close) {
    String query = "SELECT * FROM " + tableName;
    DBResults results =
      getQueryResults(driver, url, username,
                      password, query, close);
    printTableData(tableName, results, entryWidth, true);
  }

  /** Wywietla ca zawarto tabeli. Kady element zostanie 
   *  wywietlony w kolumnie o szerokoci "entryWidth" znakw,
   *  a zatem naley poda warto, ktra bdzie co najmniej rwna
   *  dugoci najduszego acucha znakw.
   */

  public static void printTable(Connection connection,
                                String tableName,
                                int entryWidth,
                                boolean close) {
    String query = "SELECT * FROM " + tableName;
    DBResults results =
      getQueryResults(connection, query, close);
    printTableData(tableName, results, entryWidth, true);
  }

  public static void printTableData(String tableName,
                                    DBResults results,
                                    int entryWidth,
                                    boolean printMetaData) {
    if (results == null) {
      return;
    }
    if (printMetaData) {
      System.out.println("Baza danych: " +
                         results.getProductName());
      System.out.println("Wersja: " +
                         results.getProductVersion());
      System.out.println();
    }
    System.out.println(tableName + ":");
    String underline =
      padString("", tableName.length()+1, "=");        
    System.out.println(underline);
    int columnCount = results.getColumnCount();
    String separator =
      makeSeparator(entryWidth, columnCount);
    System.out.println(separator);
    String row = makeRow(results.getColumnNames(), entryWidth);
    System.out.println(row);
    System.out.println(separator);
    int rowCount = results.getRowCount();
    for(int i=0; i<rowCount; i++) {
      row = makeRow(results.getRow(i), entryWidth);
      System.out.println(row);
    }
    System.out.println(separator);
  }
  
  // acuch znakw postaci "|  xxx |  xxx |  xxx |"

  private static String makeRow(String[] entries,
                                int entryWidth) {
    String row = "|";
    for(int i=0; i<entries.length; i++) {
      row = row + padString(entries[i], entryWidth, " ");
      row = row + " |";
    }
    return(row);
  }
    
  // acuch znakw postaci "+------+------+------+"
  
  private static String makeSeparator(int entryWidth,
                                      int columnCount) {
    String entry = padString("", entryWidth+1, "-");
    String separator = "+";
    for(int i=0; i<columnCount; i++) {
      separator = separator + entry + "+";
    }
    return(separator);
  }

  private static String padString(String orig, int size,
                                  String padChar) {
    if (orig == null) {
      orig = "<null>";
    }
    // Uywa obiektu StringBuffer, a nie wielokrotnej konkatenacji
    // acuchw znakw, aby unikn tworzenia zbyt wielu tymczasowych
    // obiektw klasy String..
    StringBuffer buffer = new StringBuffer("");
    int extraChars = size - orig.length();
    for(int i=0; i<extraChars; i++) {
      buffer.append(padChar);
    }
    buffer.append(orig);
    return(buffer.toString());
  }
}
