package coreservlets;

import java.sql.*;
import java.util.*;

/** Klasa do stworzenia puli pocze JDBC, ich wielkrotnego
 *  wykorzystywania i zarzdzania nimi.
 *  <P>
 *  Przykady z ksiki Java Servlet i JavaServer Pages
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2000 Marty Hall; mona kopiowa i modyfikowa bez ogranicze.
 */

public class ConnectionPool implements Runnable {
  private String driver, url, username, password;
  private int maxConnections;
  private boolean waitIfBusy;
  private Vector availableConnections, busyConnections;
  private boolean connectionPending = false;

  public ConnectionPool(String driver, String url,
                        String username, String password,
                        int initialConnections,
                        int maxConnections,
                        boolean waitIfBusy)
      throws SQLException {
    this.driver = driver;
    this.url = url;
    this.username = username;
    this.password = password;
    this.maxConnections = maxConnections;
    this.waitIfBusy = waitIfBusy;
    if (initialConnections > maxConnections) {
      initialConnections = maxConnections;
    }
    availableConnections = new Vector(initialConnections);
    busyConnections = new Vector();
    for(int i=0; i<initialConnections; i++) {
      availableConnections.addElement(makeNewConnection());
    }
  }
  
  public synchronized Connection getConnection()
      throws SQLException {
    if (!availableConnections.isEmpty()) {
      Connection existingConnection =
        (Connection)availableConnections.lastElement();
      int lastIndex = availableConnections.size() - 1;
      availableConnections.removeElementAt(lastIndex);
      // Jeli poczenie z listy dostpnych pocze bdzie 
      // zamknite (np.: gdy upyn limit czasu istnienia 
      // poczenia), to naley usun je z listy dostpnych
      // pocze i powtrzy cay proces pobierania poczenia.
      // Naley take uaktywni wszystkie wtki oczekujce na
      // poczenie ze wzgldu na przekroczenie limitu iloci
      // pocze (maxConnection).
      if (existingConnection.isClosed()) {
        notifyAll(); // wolne miejsce dla oczekujcych wtkw
        return(getConnection()); // powtrz proces
      } else {
        busyConnections.addElement(existingConnection);
        return(existingConnection);
      }
    } else {
      
      // Trzy moliwe przypadki:
      // 1) Nie zosta osignity limit iloci pocze 
      //    (maxConnections). A zatem nawizujemy poczenie 
      //    w tle jeli aktualnie jakie poczenie nie jest 
      //    nawizywane; a nastpnie czekamy na nastpne dostpne
      //    poczenie (niezalenie od tego czy byo to nowe poczenie, 
      //    czy te poczenie utworzone ju wczeniej).
      // 2) Zosta osignity limit iloci pocze (maxConnections) 
      //    oraz flaga waitIfBusy ma warto false. W takim przypadku
      //    naley zgosi wyjtek SQLException.
      // 3) Zosta osignity limit iloci pocze (maxConnections)
      //    oraz flaga waitIfBusy ma warto true. W takim przypadku 
      //    naley wykona te same czynnoci co w drugiej czci 
      //    punktu 1) - poczeka a jakie poczenie bdzie dostpne.
      
      if ((totalConnections() < maxConnections) &&
          !connectionPending) {
        makeBackgroundConnection();
      } else if (!waitIfBusy) {
        throw new SQLException("Przekroczono limit iloci pocze");
      }
      // Poczekaj na nawizanie nowego poczenia
      // (jeli zostaa wywoana metoda makeBackgroundConnection) 
      // lub na udostpnienie jakiego ju istniejcego poczenia.
      try {
        wait();
      } catch(InterruptedException ie) {}
      // kto zwolni poczenie - sprbuj moe bdzie dostpne.
      return(getConnection());
    }
  }

  // Nie mona stworzy nowego poczenia jeli adne poczenia
  // nie s dostpne, gdy w przypadku korzystania z wolnego 
  // poczenia sieciowego moe to zabra nawet kilka sekund.
  // Zamiast tego uruchom nowy wtek, ktry nawie nowe
  // poczenie z baz danych, a nastpnie poczekaj. 
  // Dziaanie wtku zostanie wznowione jeli zostanie 
  // udostpnione nowe poczenie, lub jeli bdzie mona
  // uy jakiego ju istniejcego poczenia.

  private void makeBackgroundConnection() {
    connectionPending = true;
    try {
      Thread connectThread = new Thread(this);
      connectThread.start();
    } catch(OutOfMemoryError oome) {
      // Przerwij tworzenie nowego poczenia
    }
  }

  public void run() {
    try {
      Connection connection = makeNewConnection();
      synchronized(this) {
        availableConnections.addElement(connection);
        connectionPending = false;
        notifyAll();
      }
    } catch(Exception e) { // Wyjtek SQLException lub 
      // OutOfMemory. Przerwij tworzenie nowego poczenia 
      // i poczekaj a zostanie udostpnione ktre z ju 
      // istniejcych pocze.
    }
  }

  // Ta metoda jawnie tworzy nowe poczenie. W czasie 
  // inicjalizacji obiektu ConnectionPool wywoywana jest
  // normalnie, a podczas korzystania z puli - jest 
  // wywoywana w tle.
  
  private Connection makeNewConnection()
      throws SQLException {
    try {
      // Zaaduj sterownik bazy danych, jeli jeszcze
      // nie zosta zaadowany.
      Class.forName(driver);
      // Nawi poczenie sieciowe z baz danych.
      Connection connection =
        DriverManager.getConnection(url, username, password);
      return(connection);
    } catch(ClassNotFoundException cnfe) {
      // Upraszczam blok try/catch dla osb korzystajcych z
      // tego kodu i obsuguj tylko jeden typ wyjtkw.
      throw new SQLException("Nie mona znale klasy dla sterownika: " +
                             driver);
    }
  }

  public synchronized void free(Connection connection) {
    busyConnections.removeElement(connection);
    availableConnections.addElement(connection);
    // Uaktywnij wszystkie wtki oczekujce na poczenie.
    notifyAll(); 
  }
    
  public synchronized int totalConnections() {
    return(availableConnections.size() +
           busyConnections.size());
  }

  /** Metoda zamyka wszystkie poczenia. Uywaj jej
   *  z du ostronoci: przed jej wywoaniem naley
   *  upewni si, e adne poczenia nie s w danej 
   *  chwili uywane. Zauwa, e <I>nie musisz</I> uywa
   *  tej metody gdy ju zakoczysz uywanie obiektu 
   *  ConnectionPool, gdy system gwarantuje, e wszystkie
   *  poczenia zostan zamknite podczas automatycznego
   *  czyszczenia pamici. Jednak metoda ta daje wiksz
   *  kontrol nad tym, kiedy poczenia bd zamykane.
   */

  public synchronized void closeAllConnections() {
    closeConnections(availableConnections);
    availableConnections = new Vector();
    closeConnections(busyConnections);
    busyConnections = new Vector();
  }

  private void closeConnections(Vector connections) {
    try {
      for(int i=0; i<connections.size(); i++) {
        Connection connection =
          (Connection)connections.elementAt(i);
        if (!connection.isClosed()) {
          connection.close();
        }
      }
    } catch(SQLException sqle) {
      // Zignoruj bdy, i tak wszystko bdzie 
      // usunite z pamici
    }
  }
  
  public synchronized String toString() {
    String info =
      "ConnectionPool(" + url + "," + username + ")" +
      ", dostpnych=" + availableConnections.size() +
      ", uywanych=" + busyConnections.size() +
      ", limit=" + maxConnections;
    return(info);
  }
}
