package coreservlets;

import java.io.*;
import java.util.*;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import coreservlets.beans.*;

/** Rozbudowana wersja klasy <CODE>DriverUtilities</CODE> 
 *  uzupełniona o możliwości wczytywania informacji o sterownikach
 *  baz danych innych typów z pliku XML (który domyślnie
 *  nosi nazwę drivers.xml). Do odczytu zawartości pliku używane 
 *  są zarówno DOM jak i JAXP. Plik XML ma następujący format:
 *  <P>
 *  <PRE>
 *    &lt;drivers&gt;
 *      &lt;driver&gt;
 *        &lt;vendor&gt;ORACLE&lt;/vendor&gt;
 *        &lt;description&gt;Oracle&lt;/description&gt;
 *        &lt;driver-class&gt;
 *          oracle.jdbc.driver.OracleDriver
 *        &lt;/driver-class&gt;
 *        &lt;url&gt;
 *          jdbc:oracle:thin:@[$host]:1521:[$dbName]
 *        &lt;/url&gt;
 *      &lt;/driver&gt;
 *      ...
 *    &lt;drivers&gt;
 *  </PRE>
 *  <P>
   *  Adres URL powinien zawierać symbole specjalne
   *  [$host] oraz [$dbName], które w metodzie 
   *  <CODE>makeURL</CODE> zostaną zamienione 
   *  odpowiednio na <I>nazwę komputera</I>
   *  oraz <I>nazwę bazy danych</I>.
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall & Larry Brown; można kopiować i modyfikować bez ograniczeń.
 */

public class DriverUtilities2 extends DriverUtilities {
  public static final String DEFAULT_FILE = "drivers.xml";

  /** Wczytanie informacji o sterowniku z domyślnego 
   *  pliku XML.
   */

  public static void loadDrivers() {
    DriverUtilities2.loadDrivers(DEFAULT_FILE);
  }

  /** Wczytanie informacji o sterowniku z podanego pliku 
   *  XML. Każdy typ bazy jest reprezentowany przez obiekt
   *  typu <CODE>DriverInfoBean</CODE>; wszystkie te obiekty
   *  są z kolei przechowywane w mapie, której kluczami są
   *  typy baz. Metody należy używać w przypadkach gdy 
   *  konieczne jest wczytanie informacji o sterownikach 
   *  z innego pliku niż domyślny - drivers.xml.
   */

  public static void loadDrivers(String filename) {
    File file = new File(filename);
    try {
      InputStream in = new FileInputStream(file);
      DocumentBuilderFactory builderFactory =
        DocumentBuilderFactory.newInstance();
      DocumentBuilder builder =
        builderFactory.newDocumentBuilder();
      Document document = builder.parse(in);
      document.getDocumentElement().normalize();
      Element rootElement = document.getDocumentElement();
      NodeList driverElements =
        rootElement.getElementsByTagName("driver");
      // Stworzenie obiektu DriverInfoBean dla każdego typu bazy
      for(int i=0; i<driverElements.getLength(); i++) {
        Node node = driverElements.item(i);
        DriverInfoBean info =
          DriverUtilities2.createDriverInfoBean(node);
        if (info != null) {
          addDriverInfoBean(info);
        }
      }
    } catch(FileNotFoundException fnfe) {
      System.err.println("Nie można znaleźć pliku " + filename);
    } catch(IOException ioe) {
      System.err.println("Nie można odczytać zawartości pliku: " + ioe);
    } catch(ParserConfigurationException pce) {
      System.err.println("Nie można utworzyć obiektu DocumentBuilder");
    } catch(SAXException se) {
      System.err.println("Wystąpiły problemy podczas analizy zawartośći pliku: " + se);
    }
  }

  /** Utworzenie obiektu DriverInfoBean na podstawie 
   *  informacji zapisanych w węźle DOM o następującym formacie:
   *  <P>
   *  <PRE>
   *    &lt;driver&gt;
   *      &lt;vendor&gt;ORACLE&lt;/vendor&gt;
   *      &lt;description&gt;Oracle&lt;/description&gt;
   *      &lt;driver-class&gt;
   *         oracle.jdbc.driver.OracleDriver
   *      &lt;/driver-class&gt;
   *      &lt;url&gt;
   *        jdbc:oracle:thin:@[$host]:1521:[$dbName]
   *      &lt;/url&gt;
   *    &lt;/driver&gt;
   *  </PRE>
   */

  public static DriverInfoBean createDriverInfoBean(Node node) {
    Map map = new HashMap();
    NodeList children = node.getChildNodes();
    for(int i=0; i<children.getLength(); i++) {
      Node child = children.item(i);
      String nodeName = child.getNodeName();
      if (child instanceof Element) {
        Node textNode = child.getChildNodes().item(0);
        if (textNode != null) {
          map.put(nodeName, textNode.getNodeValue());
        }
      }
    }
    return(new DriverInfoBean((String)map.get("vendor"),
                              (String)map.get("description"),
                              (String)map.get("driver-class"),
                              (String)map.get("url")));
  }
}           