package coreservlets;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.util.*;

/** Serwlet obsługuje żądania generowania strony zawierającej numPrimes
 *  liczb pierwszych, z których każda ma numDigits cyfr.
 *  Obliczenia wykonywane są przez wątek o niskim priorytecie
 *  zwracający wyłącznie obliczenia, które w danej chwili
 *  zostały już zakończone. Jeśli wyniki nie są kompletne, 
 *  serwlet generuje nagłówek odpowiedzi Refresh, informując
 *  przeglądarkę, że należy powtórnie zażądać wyników po 
 *  pewnym czasie. Serwlet przechowuje także krótką listę
 *  ostatnio wygenerowanych liczb pierwszych, które zwraca 
 *  natychmiast po otrzymaniu żądania o takich samych parametrach n i m,
 *  jakie były wykorzystane w jednym z ostatnio zakończonych 
 *  obliczeń.
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall; można kopiować i modyfikować bez ograniczeń.
 */

public class PrimeNumberServlet extends HttpServlet {
  private ArrayList primeListCollection = new ArrayList();
  private int maxPrimeLists = 30;
  
  public void doGet(HttpServletRequest request,
                    HttpServletResponse response)
      throws ServletException, IOException {
    int numPrimes =
      ServletUtilities.getIntParameter(request,
                                       "numPrimes", 50);
    int numDigits =
      ServletUtilities.getIntParameter(request,
                                       "numDigits", 120);
    PrimeList primeList =
      findPrimeList(primeListCollection, numPrimes, numDigits);
    if (primeList == null) {
      primeList = new PrimeList(numPrimes, numDigits, true);
      // Wiele żądań to tego serwletu współdzieli te
      // same zmienne instancyjne (pola) klasy PrimeNumbers.
      // Dlatego trzeba synchronizować wszystkie próby dostępu do
      // pól serwletu.
      synchronized(primeListCollection) {
        if (primeListCollection.size() >= maxPrimeLists)
          primeListCollection.remove(0);
        primeListCollection.add(primeList);
      }
    }
    ArrayList currentPrimes = primeList.getPrimes();
    int numCurrentPrimes = currentPrimes.size();
    int numPrimesRemaining = (numPrimes - numCurrentPrimes);
    boolean isLastResult = (numPrimesRemaining == 0);
    if (!isLastResult) {
      response.setIntHeader("Refresh", 5);
    }
    response.setContentType("text/html; charset=ISO-8859-2");
    PrintWriter out = response.getWriter();
    String title = "Kilka " + numDigits + "-cyfrowych liczb pierwszych";
    out.println(ServletUtilities.headWithTitle(title) +
                "<BODY BGCOLOR=\"#FDF5E6\">\n" +
                "<H2 ALIGN=CENTER>" + title + "</H2>\n" +
                "<H3>Odnalezione liczby pierwsze o długości " + numDigits +
                " cyfr lub większej: " + numCurrentPrimes +
                ".</H3>");
    if (isLastResult)
      out.println("<B>Obliczenia zakończone.</B>");
    else
      out.println("<B>Ciągle szukam " + numPrimesRemaining +
                  " pozostałych liczb<BLINK>...</BLINK></B>");
    out.println("<OL>");
    for(int i=0; i<numCurrentPrimes; i++) {
      out.println("  <LI>" + currentPrimes.get(i));
    }
    out.println("</OL>");
    out.println("</BODY></HTML>");
  }

  /* Metoda sprawdza obecność aktualnie obliczanej
   * lub już obliczonej listy liczb pierwszych o podanej
   * długości oraz liczbie cyfr w liczbie. Jeśli taka lista 
   * istnieje, serwlet zwraca tę listę bez uruchamiania nowego
   * wątku obliczeniowego działającego w tle. Lista wyników
   * powinna być możliwie niewielka, aby nie zabierała zbyt 
   * dużo pamięci na serwerze. Dostęp do tej listy należy
   * synchronizować, gdyż jednocześnie może się do niej 
   * odwoływać wiele wątków.
   */
  
  private PrimeList findPrimeList(ArrayList primeListCollection,
                                  int numPrimes,
                                  int numDigits) {
    synchronized(primeListCollection) {
      for(int i=0; i<primeListCollection.size(); i++) {
        PrimeList primes =
          (PrimeList)primeListCollection.get(i);
        if ((numPrimes == primes.numPrimes()) &&
            (numDigits == primes.numDigits()))
          return(primes);
      }
      return(null);
    }
  }
}
