package coreservlets;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.io.*;
import javax.imageio.*;

/** Klasa pomocnicza ułatwia generowanie obrazów 
 *  prezentujących komunikat z cieniem.
 *  <P>
 *	Wymaga zastosowania JDK 1.4 gdyż używa klasy ImageIO.
 *	JDK 1.4 jest standardowo używany w serwerach aplikacji
 *	zgodnych z platformą J2EE udostępniającą technologie
 *	JavaServlet 2.4 oraz JSP 2.0. Jednak niezależne
 *	mechanizmy obsługi serwletów/JSP wymagają zastosowania
 *	jedynie JDK w wersji 1.3 lub nowszej; natomiast 
 *	specyfikacja serwletów 2.2 wymaga jedynie JDK 1.2 
 *  (lub nowszej wersji). A zatem, chociaż większość
 *  serwerów używa JDK 1.4, to nie ma gwarancji, że
 *  poniższy serwlet będzie działać na wszystkich 
 *  dostępnych serwerach.
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall; można kopiować i modyfikować bez ograniczeń.
 */

public class MessageImage {
  
  /** Metoda tworzy obraz przedstawiający podany
   *  łańcuch znaków z ukośnym cieniem. Jest ona
   *  używana w serwlecie ShadowedText.
   */

  public static BufferedImage makeMessageImage(String message,
                                               String fontName,
                                               int fontSize) {
    
    Font font = new Font(fontName, Font.PLAIN, fontSize);
    FontMetrics metrics = getFontMetrics(font);
    int messageWidth = metrics.stringWidth(message);
    int baselineX = messageWidth/10;
    int width = messageWidth+2*(baselineX + fontSize);
    int height = fontSize*7/2;
    int baselineY = height*8/10;
    BufferedImage messageImage =
      new BufferedImage(width, height,
                        BufferedImage.TYPE_INT_RGB);
    Graphics2D g2d = (Graphics2D)messageImage.getGraphics();
    g2d.setBackground(Color.white);
    g2d.clearRect(0, 0, width, height);
    g2d.setFont(font);
    g2d.translate(baselineX, baselineY);
    g2d.setPaint(Color.lightGray);
    AffineTransform origTransform = g2d.getTransform();
    g2d.shear(-0.95, 0);
    g2d.scale(1, 3);
    g2d.drawString(message, 0, 0);
    g2d.setTransform(origTransform);
    g2d.setPaint(Color.black);
    g2d.drawString(message, 0, 0);
    return(messageImage);
  }
  
  public static void writeJPEG(BufferedImage image,
                               OutputStream out) {
    try {
      ImageIO.write(image, "jpg", out);
    } catch(IOException ioe) {
      System.err.println("Błąd podczas zapisu obrazu JPEG: " + ioe);
    }
  }

  public static void writeJPEG(BufferedImage image,
                               File file) {
    try {
      ImageIO.write(image, "jpg", file);
    } catch(IOException ioe) {
      System.err.println("Błąd podczas zapisu pliku JPEG: " + ioe);
    }
  }

  public static String[] getFontNames() {
    GraphicsEnvironment env =
      GraphicsEnvironment.getLocalGraphicsEnvironment();
    return(env.getAvailableFontFamilyNames());
  }
  
  /** Do uzyskania dostępu do obiektu FontMetics konieczny 
   *  jest obiekt Graphics. Obiekt FontMetrics określa wielkość
   *  łańcucha znaków wyświetlonego określoną czcionką. Jednak 
   *  do uzyskania obiektu Graphics konieczny jest obiekt 
   *  obrazu. Ponieważ wielkość wynikowego obrazu będzie 
   *  zależeć od wielkości prezentowanego łańcucha znaków,
   *  w pierwszej kolejności tworzony jest bardzo mały
   *  obraz tymczasowy, który służy do pobrania obiektu 
   *  FontMetrics i określenia wielkości wynikowego obrazu.
   *  Dopiero potem tworzony jest prawdziwy obraz o wyznaczonych
   *  wcześniej wymiarach.
   */
  
  private static FontMetrics getFontMetrics(Font font) {
    BufferedImage tempImage =
      new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
    Graphics2D g2d = (Graphics2D)tempImage.getGraphics();
    return(g2d.getFontMetrics(font));
  }
}
