package coreservlets;

import java.sql.*;
import coreservlets.beans.*;

/** Przykład umożliwiający przetestowanie różnic w czasie 
 *  wykonania zapytania SQL pomiędzy zastosowaniem 
 *  nieprzetworzonych zapytań i zapytań przygotowanych.
 *  Uzyskane wyniki w bardzo dużym stopniu zależą od 
 *  używanego sterownika i serwera bazy danych, a ich 
 *  wartości mogą się znacząco różnić. Przy użytej
 *  konfiguracji systemu, wykonanie przygotowanych zapytań
 *  w bazie danych Oracle9i zajmowało 62% czasu koniecznego do
 *  wykonania tych samych zapytań w wersji nieprzetworzonej;
 *  W przypadku wykorzystania serwera MySQL, czas konieczny 
 *  do wykonania tych samych zapytań przygotowanych był 
 *  jedynie o 8% krótszy od czasu koniecznego do wykonania 
 *  zapytań nieprzetworzonych.
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall & Larry Brown; można kopiować i modyfikować bez ograniczeń.
 */

public class PreparedStatements {
  public static void main(String[] args) {
    if (args.length < 5) {
      printUsage();
      return;
    }
    String vendor = args[4];
    // W celu wczytania informacji o sterownikach z pliku 
    // XML należy użyć metody 
    // DriverUtilities2.loadDrivers().
    DriverUtilities.loadDrivers();
    if (!DriverUtilities.isValidVendor(vendor)) {
      printUsage();
      return;
    }
    String driver = DriverUtilities.getDriver(vendor);
    String host = args[0];
    String dbName = args[1];
    String url =
      DriverUtilities.makeURL(host, dbName, vendor);
    String username = args[2];
    String password = args[3];
    // Metoda "print" została użyta tylko po to by potwierdzić, 
    // że wszystko jest OK nie należy jej używać
    // podczas określania czasów wykonywania zapytań.
    boolean print = false;
    if ((args.length > 5) && (args[5].equals("print"))) {
      print = true;
    }
    Connection connection =
      ConnectionInfoBean.getConnection(driver, url,
                                       username, password);
    if (connection != null) {
      doPreparedStatements(connection, print);
      doRawQueries(connection, print);
    }
    try {
      connection.close();
    } catch(SQLException sqle) {
      System.err.println("Wystąpiły problemy podczas zamykania połączenia: " + sqle);
    }
  }

  private static void doPreparedStatements(Connection conn,
                                           boolean print) {
    try {
      String queryFormat =
        "SELECT id FROM music WHERE price < ?";
      PreparedStatement statement =
        conn.prepareStatement(queryFormat);
      long startTime = System.currentTimeMillis();
      for(int i=0; i<100; i++) {
        statement.setFloat(1, i/4);
        ResultSet results = statement.executeQuery();
        if (print) {
          showResults(results);
        }
      }
      long stopTime = System.currentTimeMillis();
      double elapsedTime = (stopTime - startTime)/1000.0;
      System.out.println("Wykonanie polecenia przygotowanego " +
                         "100 razy zajęło " +
                         elapsedTime + " sekund.");
    } catch(SQLException sqle) {
      System.err.println("Błąd podczas wykonywania polecenia: " + sqle);
    }
  }

  public static void doRawQueries(Connection conn,
                                  boolean print) {
    try {
      String queryFormat =
        "SELECT id FROM music WHERE price < ";
      Statement statement = conn.createStatement();
      long startTime = System.currentTimeMillis();
      for(int i=0; i<100; i++) {
        ResultSet results =
          statement.executeQuery(queryFormat + i/4);
        if (print) {
          showResults(results);
        }
      }
      long stopTime = System.currentTimeMillis();
      double elapsedTime = (stopTime - startTime)/1000.0;
      System.out.println("Wykonanie polecenia " +
                         "100 razy zajęło " +
                         elapsedTime + " sekund.");
    } catch(SQLException sqle) {
      System.err.println("Błąd podczas wykonywania polecenia: " + sqle);
    }
  }

  private static void showResults(ResultSet results)
      throws SQLException {
    while(results.next()) {
      System.out.print(results.getString(1) + " ");
    }
    System.out.println();
  }

  private static void printUsage() {
    System.out.println("Sposób użycia: PreparedStatements host " +
                       "dbName username password " +
                       "vendor [print].");
  }
}          