package coreservlets;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;
import coreservlets.beans.*;

/** Przykład uproszczonego sposobu obsługi danych. Prezentuje 
 *  dwa rozwiązania:
 *  <OL>
 *    <LI>Automatyczne zapisywanie w komponencie informacji 
 *        pobranych z ż±dania.
 *    <LI>Użycie tego samego serwletu zarówno do wygenerowania
 *        formularza służącego do wprowadzania danych, jak 
 *        i do przetwarzania wyników. Dzięki temu, jeśli 
 *        w formularzu zostaną pominięte jakieś niezbędne pola
 *        serwlet będzie w stanie ponownie wyswietlić formularz
 *        i wypełnić go informacjami które użytkownik wcześniej
 *        podał.
 *  </UL>
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall; można kopiować i modyfikować bez ograniczeń.
 */

public class BidServlet extends HttpServlet {

  /** Metoda próbuje wypełnić komponent reprezentuj±cy
   *  informacje wpisane przez użytkownika w formularzu.
   *  Jeśli dane są kompletne, metoda wyświetla wyniki.
   *  Jeśli żadne dane nie zostały podane lub jeśli podano 
   *  tylko niektóre informacje, to metoda ponownie 
   *  wyświetla formularz HTML.
   */
  
  public void doGet(HttpServletRequest request,
                    HttpServletResponse response)
      throws ServletException, IOException {
    BidInfo bid = new BidInfo();
    BeanUtilities.populateBean(bid, request);
    if (bid.isComplete()) {
      // Wszystkie niezbędne informacje zostały podane
      // wyświetlamy wyniki
      showBid(request, response, bid);
    } else {
      // Brak danych lub podano niekompletne informacje
      // ponownie wyświetlamy formularz
      showEntryForm(request, response, bid);
    }
  }

  /** Wszystkie wymagane informacje zostały podane, można
   *  generować stronę wynikową */
  
  private void showBid(HttpServletRequest request,
                       HttpServletResponse response,
                       BidInfo bid)
      throws ServletException, IOException {
    submitBid(bid);
    response.setContentType("text/html; charset=ISO-8859-2");
    PrintWriter out = response.getWriter();
    String title = "Oferta została zarejestrowana";
    out.println
      (DOCTYPE +
       "<HTML>\n" +
       "<HEAD><TITLE>" + title + "</TITLE></HEAD>\n" +
       "<BODY BGCOLOR=\"#FDF5E6\"><CENTER>\n" +
       "<H1>" + title + "</H1>\n" +
       "Twoja oferta została uaktywniona. Jeśli wygra licytację,\n" +
       "to zostaniesz o tym poinformowany do 24 godzin po zakończeniu\n" +
       "licytacji.\n" +
       "<P>\n" +
       "<TABLE BORDER=1>\n" +
       "  <TR><TH BGCOLOR=\"BLACK\"><FONT COLOR=\"WHITE\">" +
       bid.getItemName() + "</FONT></TH></TR>\n" +
       "  <TR><TH>ID towaru: " +
       bid.getItemID() + "</TH></TR>\n" +
       "  <TR><TH>Personalia: " +
       bid.getBidderName() + "</TH></TR>\n" +
       "  <TR><TH>Adres email: " +
       bid.getEmailAddress() + "</TH></TR>\n" +
       "  <TR><TH>Oferowana cena: zł" +
       bid.getBidPrice() + "</TH></TR>\n" +
       "  <TR><TH>Automatyczna inkrementacja ceny: " +
       bid.isAutoIncrement() + "</TH></TR>\n" +
       "</TABLE></CENTER></BODY></HTML>");
  }

  /** Jeśli nie podano żadnych informacji, to metoda ta 
   *  wyświetla pusty formularz. Jeśli użytkownik podał jakieś 
   *  informacje, to metoda wyświetli formularz umieszczając 
   *  w odpowiednich polach podane wcześniej wartości oraz 
   *  prosi użytkownika o uzupełnienie pozostałych pól.
   */

  private void showEntryForm(HttpServletRequest request,
                      HttpServletResponse response,
                      BidInfo bid)
      throws ServletException, IOException {
    boolean isPartlyComplete = bid.isPartlyComplete();
    response.setContentType("text/html; charset=ISO-8859-2");
    PrintWriter out = response.getWriter();
    String title =
      "Witamy na witrynie Automatyczne-Licytacje.";
    out.println
      (DOCTYPE +
       "<HTML>\n" +
       "<HEAD><TITLE>" + title + "</TITLE></HEAD>\n" +
       "<BODY BGCOLOR=\"#FDF5E6\"><CENTER>\n" +
       "<H1>" + title + "</H1>\n" +
       warning(isPartlyComplete) +
       "<FORM>\n" +
       inputElement("ID towaru", "itemID",
                    bid.getItemID(), isPartlyComplete) +
       inputElement("Nazwa towaru", "itemName",
                    bid.getItemName(), isPartlyComplete) +
       inputElement("Twoje personalia", "bidderName",
                    bid.getBidderName(), isPartlyComplete) +
       inputElement("Twój adres email", "emailAddress",
                    bid.getEmailAddress(), isPartlyComplete) +
       inputElement("Oferowana cena", "bidPrice",
                    bid.getBidPrice(), isPartlyComplete) +
       checkbox("Czy automatycznie inkremetować cenę by przebijać inne oferty?",
                "autoIncrement", bid.isAutoIncrement()) +
       "<INPUT TYPE=\"SUBMIT\" VALUE=\"Prześlij ofertę\"></FORM>\n" +
       "</CENTER></BODY></HTML>");
  }

  private void submitBid(BidInfo bid) {
  	// Kod specyficzny dla tej aplikacji, służący do rejestracji
  	// oferty. Najważniejsze w tej metodzie jest to, iż
  	// przekazujemy do niej faktyczny obiekt zawierający
  	// kompletne informacje, a nie grupę łańcuchów znaków.
  }

  private String warning(boolean isFormPartlyComplete) {
    if(isFormPartlyComplete) {
      return("<H2>Brak wymaganych informacji! " +
             "Uzupełnij je i ponownie wyślij formularz.</H2>\n");
    } else {
      return("");
    }
  }

  /** Metoda tworzy pole tekstowe poprzedzone etykietą.
   *  Jeśli to konkretne pole nie zostało podane lecz 
   *  formularz został częściowo wypełniony, to metoda 
   *  wyświetla dodatkowy komentarz informujący użytkownika 
   *  o tym, iż pole jest wymagane.
   */
  
  private String inputElement(String prompt,
                              String name,
                              String value,
                              boolean shouldPrompt) {
    String message = "";
    if (shouldPrompt && ((value == null) || value.equals(""))) {
      message = "<B>To pole jest wymagane!</B> ";
    }
    return(message + prompt + ": " +
           "<INPUT TYPE=\"TEXT\" NAME=\"" + name + "\"" +
           " VALUE=\"" + value + "\"><BR>\n");
  }

  private String inputElement(String prompt,
                              String name,
                              double value,
                              boolean shouldPrompt) {
    String num;
    if (value == 0.0) {
      num = "";
    } else {
      num = String.valueOf(value);
    }
    return(inputElement(prompt, name, num, shouldPrompt));
  }

  private String checkbox(String prompt,
                          String name,
                          boolean isChecked) {
    String result =
      prompt + ": " +
      "<INPUT TYPE=\"CHECKBOX\" NAME=\"" + name + "\"";
    if (isChecked) {
      result = result + " CHECKED";
    }
    result = result + "><BR>\n";
    return(result);
  }

  private final String DOCTYPE =
    "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 " +
    "Transitional//EN\">\n";
}
