package coreservlets;

import java.sql.*;

/** Program wykonuje następujące testy na bazie danych:
 *  <OL>
 *  <LI>Tworzy połączenie JDBC z bazą i wyświetla jej nazwę
 *      oraz numer wersji.
 *  <LI>Tworzy prostą tabelą "authors" zawierającą 
 *      identyfikatory, imiona oraz nazwiska autorów
 *      niniejszej książki.
 *  <LI>Odczytuje wszystkie wiersze tabeli authors.
 *  <LI>Określa wersję JDBC z jaką jest zgodny sterownik.
 *      Informacji tej należy używać z ostrożnością:
 *      Samo wyświetlenie tych informacji nie oznacza wcale,
 *      że sterownik posiada certyfikat zgodności z tą 
 *      wersją JDBC.
 *  </OL>
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall & Larry Brown; można kopiować i modyfikować bez ograniczeń.
 */

public class TestDatabase {
  private String driver;
  private String url;
  private String username;
  private String password;

  public TestDatabase(String driver, String url,
                      String username, String password) {
    this.driver = driver;
    this.url = url;
    this.username = username;
    this.password = password;
  }

  /** Test połączenia JDBC z bazą danych i wyświetlenie
   *  jej nazwy oraz numeru wersji.
   */

  public void testConnection() {
    System.out.println();
    System.out.println("Testowanie połączenia z bazą danych...\n");
    Connection connection = getConnection();
    if (connection == null) {
      System.out.println("Test nieudany.");
      return;
    }
    try {
      DatabaseMetaData dbMetaData = connection.getMetaData();
      String productName =
        dbMetaData.getDatabaseProductName();
      String productVersion =
        dbMetaData.getDatabaseProductVersion();
      String driverName = dbMetaData.getDriverName();
      String driverVersion = dbMetaData.getDriverVersion();
      System.out.println("Sterownik: " + driver);
      System.out.println("URL: " + url);
      System.out.println("Nazwa użytkownika: " + username);
      System.out.println("Hasło: " + password);
      System.out.println("Nazwa produktu: " + productName);
      System.out.println("Wersja produktu: " + productVersion);
      System.out.println("Nazwa sterownika: " + driverName);
      System.out.println("Wersja sterownika: " + driverVersion);
    } catch(SQLException sqle) {
      System.err.println("Błąd podczas nawiązywania połączenia: " + sqle);
    } finally {
      closeConnection(connection);
    }
    System.out.println();
  }

  /** Metoda, która tworzy prostą tabelę (authors) zawierającą
   *  kolumny: id, first_name oraz last_name. W tabeli zapisywane
   *  są personalia dwóch autorów niniejszej książki.
   */

  public void createTable() {
    System.out.print("Tworzenie tabeli authors ... ");
    Connection connection = getConnection();
    if (connection == null) {
      System.out.println("zakończone niepowodzeniem");
      return;
    }
    try {
      String format =
        "(id INTEGER, first_name VARCHAR(12), " +
        " last_name VARCHAR(12))";
      String[] rows = { "(1, 'Marty', 'Hall')",
                        "(2, 'Larry', 'Brown')" };
      Statement statement = connection.createStatement();
      // Usunięcie istniejącej tabeli (jeśli jest), bez 
      // zwracania błędów jeśli jej nie ma. Właśnie po to
      // został zastosowany osobny blok try-catch.
      try {
        statement.execute("DROP TABLE authors");
      } catch(SQLException sqle) {}
      String createCommand =
        "CREATE TABLE authors " + format;
      statement.execute(createCommand);
      String insertPrefix =
        "INSERT INTO authors VALUES";
      for(int i=0; i<rows.length; i++) {
        statement.execute(insertPrefix + rows[i]);
      }
      System.out.println("zakończone pomyślnie");
    } catch(SQLException sqle) {
      System.out.println("zakończone niepowodzeniem");
      System.err.println("Błąd podczas tworzenia tabeli: " + sqle);
    } finally {
      closeConnection(connection);
    }
    System.out.println();
  }

  /** Pobranie wszystkich wierszy z tabeli "authors". */

  public void executeQuery() {
    System.out.println("Zapytanie do tabeli authors ... ");
    Connection connection = getConnection();
    if (connection == null) {
      System.out.println("Błąd.");
      return;
    }
    try {
      Statement statement = connection.createStatement();
      String query = "SELECT * FROM authors";
      ResultSet resultSet = statement.executeQuery(query);
      ResultSetMetaData resultSetMetaData =
        resultSet.getMetaData();
      int columnCount = resultSetMetaData.getColumnCount();
      // Wyświetlenie kolumn
      String[] columns = new String[columnCount];
      int[] widths = new int[columnCount];
      for(int i=1; i <= columnCount; i++) {
        columns[i-1] = resultSetMetaData.getColumnName(i);
        widths[i-1] = resultSetMetaData.getColumnDisplaySize(i);
      }
      System.out.println(makeSeparator(widths));
      System.out.println(makeRow(columns, widths));
      // Wyświetlenie wierszy
      System.out.println(makeSeparator(widths));
      String[] rowData = new String[columnCount];
      while(resultSet.next()) {
        for(int i=1; i <= columnCount; i++) {
          rowData[i-1] = resultSet.getString(i);
        }
        System.out.println(makeRow(rowData, widths));
      }
      System.out.println(makeSeparator(widths));
    } catch(SQLException sqle) {
      System.err.println("Błąd podczas wykonywania zapytania: " + sqle);
    } finally {
      closeConnection(connection);
    }
    System.out.println();
  }

  /** Metoda sprawdza wersję JDBC.
   *  Początkowo sprawdzana jest możliwość wywołania metody
   *  last() dostępnej w JDBC 2.0. Następnie podejmowana
   *  jest próba wywołania metod getJDBCMajorVersion oraz
   *  getJDBCMinorVersion dostępnych w JDBC 3.0.
   */

  public void checkJDBCVersion() {
    System.out.println();
    System.out.println("Sprawdzanie wersji JDBC ...\n");
    Connection connection = getConnection();
    if (connection == null) {
      System.out.println("Test nieudany.");
      return;
    }
    int majorVersion = 1;
    int minorVersion = 0;
    try {
      Statement statement = connection.createStatement(
                              ResultSet.TYPE_SCROLL_INSENSITIVE,
                              ResultSet.CONCUR_READ_ONLY);
      String query = "SELECT * FROM authors";
      ResultSet resultSet = statement.executeQuery(query);
      resultSet.last(); // JDBC 2.0
      majorVersion = 2;
    } catch(SQLException sqle) {
      // Ignorujemy - metoda last() nie jest dostępna
    }
    try {
      DatabaseMetaData dbMetaData = connection.getMetaData();
      majorVersion = dbMetaData.getJDBCMajorVersion(); // JDBC 3.0
      minorVersion = dbMetaData.getJDBCMinorVersion(); // JDBC 3.0
    } catch(Throwable throwable) {
      // Ignorujemy - metody nie są dostępne
    } finally {
      closeConnection(connection);
    }
    System.out.println("Wersja JDBC: " +
                       majorVersion + "." + minorVersion);
  }

  // Początek tabeli wyników "|  xxx |  xxx |  xxx |"

  private String makeRow(String[] entries, int[] widths) {
    String row = "|";
    for(int i=0; i<entries.length; i++) {
      row = row + padString(entries[i], widths[i], " ");
      row = row + " |";
    }
    return(row);
  }

  // Łańcuch znaków postaci "+------+------+------+"

  private String makeSeparator(int[] widths) {
    String separator = "+";
    for(int i=0; i<widths.length; i++) {
      separator += padString("", widths[i] + 1, "-") + "+";
    }
    return(separator);
  }

  private String padString(String orig, int size,
                           String padChar) {
    if (orig == null) {
      orig = "<null>";
    }
    // Używany jest obiekt StringBuffer a nie standardowe
    // łączenie łańcuchów znaków, aby uniknąć tworzenia 
    // zbyt wielu tymczasowych łańcuchów.
    StringBuffer buffer = new StringBuffer(padChar);
    int extraChars = size - orig.length();
    buffer.append(orig);
    for(int i=0; i<extraChars; i++) {
      buffer.append(padChar);
    }
    return(buffer.toString());
  }

  /** Pobranie nowego połączenia z bazą danych lub zwrócenie
   *  wartości null w przypadku niepowodzenia.
   */

  public Connection getConnection() {
    try {
      Class.forName(driver);
      Connection connection =
        DriverManager.getConnection(url, username,
                                    password);
      return(connection);
    } catch(ClassNotFoundException cnfe) {
      System.err.println("Błąd wczytywania sterownika: " + cnfe);
      return(null);
    } catch(SQLException sqle) {
      System.err.println("Błąd nawiązania połączenia: " + sqle);
      return(null);
    }
  }

  /** Zamknięcie połączenia z bazą danych. */

  private void closeConnection(Connection connection) {
    try {
      connection.close();
    } catch(SQLException sqle) {
      System.err.println("Błąd podczas zamykania połączenia: " + sqle);
      connection = null;
    }
  }

  public static void main(String[] args) {
    if (args.length < 5) {
      printUsage();
      return;
    }
    String vendor = args[4];
    // Aby wczytać dane sterownika z pliku XML, należy 
    // użyć metody DriverUtilities2.loadDrivers()
    DriverUtilities.loadDrivers();
    if (!DriverUtilities.isValidVendor(vendor)) {
      printUsage();
      return;
    }
    String driver = DriverUtilities.getDriver(vendor);
    String host = args[0];
    String dbName = args[1];
    String url =
      DriverUtilities.makeURL(host, dbName, vendor);
    String username = args[2];
    String password = args[3];

    TestDatabase database =
      new TestDatabase(driver, url, username, password);
    database.testConnection();
    database.createTable();
    database.executeQuery();
    database.checkJDBCVersion();
  }
  private static void printUsage() {
    System.out.println("Sposób użycia: TestDatabase komputer nazwaBazy " +
                       "nazwaUżytkownika hasło typ.");
  }
}        