package coreservlets;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.util.*;
import java.text.*;

/** Wyświetla informacje o wszystkich produktach, które aktualnie
 *  znajdują się w koszyku (ShoppingCart). Przeglądarki
 *  mają swoje własne sesje, na podstawie których 
 *  określana jest przynależność koszyków. Jeśli jest to 
 *  jest pierwsza wizyta na stronie umożliwiającej 
 *  składanie zamówień, tworzony jest nowy koszyk.
 *  Użytkownicy zazwyczaj przechodzą na tę stronę ze stron
 *  prezentujących produkty, które można zamawiać, dlatego
 *  też ta strona dodaje nowy element do koszyka. Jednak
 *  użytkownicy mogą zapamiętać adres tej strony i wyświetlać
 *  ją posługując się listą ulubionych stron. Mogą także
 *  wrócić na nią klikając przycisk "Aktualizuj zamówienie"
 *  po zmianie liczby egzemplarzy jednego z zamawianych 
 *  produktów.
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall; można kopiować i modyfikować bez ograniczeń.
 */

public class OrderPage extends HttpServlet {
  public void doGet(HttpServletRequest request,
                    HttpServletResponse response)
      throws ServletException, IOException {
    HttpSession session = request.getSession();
    ShoppingCart cart;
    synchronized(session) {
      cart = (ShoppingCart)session.getAttribute("shoppingCart");
      // Dla nowych użytkowników tworzone są nowe koszyki.
      // Użytkownicy, którzy już dysponują koszykami 
      // używają tych, które zostały dla nich wcześniej utworzone .
      if (cart == null) {
        cart = new ShoppingCart();
        session.setAttribute("shoppingCart", cart);
      }
      String itemID = request.getParameter("itemID");
      if (itemID != null) {
        String numItemsString =
          request.getParameter("numItems");
        if (numItemsString == null) {
          // Jeśli w żądaniu został podany identyfikator (itemID) ale
          // nie liczba, oznacza to, że użytkownik trafił
          // tutaj klikając przycisk "Dodaj do koszyka" na jednej
          // ze stron prezentujących produkty z katalogu.
          cart.addItem(itemID);
        } else {
          // Jeśli w żądaniu został podany zarówno identyfikator 
          // (itemID) jak i liczba, oznacza to, że użytkownik 
          // trafił na stronę klikając przycisk "Aktualizuj
          // zamówienie" po zmianie liczby egzemplarzy jednego
          // z zamawianych produktów. Należy zwrócić uwagę, iż podanie 
          // wartości 0 jako liczby egzemplarzy zamawianego produktu 
          // sprawi, że dany produkt zostanie usunięty z koszyka.
          int numItems;
          try {
            numItems = Integer.parseInt(numItemsString);
          } catch(NumberFormatException nfe) {
            numItems = 1;
          }
          cart.setNumOrdered(itemID, numItems);
        }
      }
    }
    // Pokaż status zamówienia niezależnie od tego, czy użytkownik 
    // je zmodyfikował czy nie.
    response.setContentType("text/html; charset=ISO-8859-2");
    PrintWriter out = response.getWriter();
    String title = "Status zamówienia";
    String docType =
      "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 " +
      "Transitional//EN\">\n";
    out.println(docType +
                "<HTML>\n" +
                "<HEAD><TITLE>" + title + "</TITLE></HEAD>\n" +
                "<BODY BGCOLOR=\"#FDF5E6\">\n" +
                "<H1 ALIGN=\"CENTER\">" + title + "</H1>");
    synchronized(session) {
      List itemsOrdered = cart.getItemsOrdered();
      if (itemsOrdered.size() == 0) {
        out.println("<H2><I>W koszyku nie ma żadnych towarów...</I></H2>");
      } else {
        // Jeśli w koszyku jest co najmniej jeden produkt,
        // wyświetl tabelę z informacjami o nim.
        out.println
          ("<TABLE BORDER=1 ALIGN=\"CENTER\">\n" +
           "<TR BGCOLOR=\"#FFAD00\">\n" +
           "  <TH>Identyfikator</TH><TH>Opis</TH>\n" +
           "  <TH>Cena jednostkowa</TH><TH>Ilość</TH><TH>Wartość</TH>");
        ItemOrder order;
        // Zaokrąglamy do dwóch miejsc po przecinku, 
        // wstawiamy znak dolara (lub innej waluty) itd.
        // wszystko zgodnie z bieżącymi ustawieniami lokalnymi.
        NumberFormat formatter =
          NumberFormat.getCurrencyInstance();
        // Dla każdego produktu umieszczonego w koszyku
        // tworzymy wiersz tabeli zawierający identyfikator
        // produktu (itemID), opis, jego cenę jednostkową,
        // liczbę zamówionych egzemplarzy oraz łączną cenę.
        // Liczbę zamawianych egzemplarzy wyświetlamy
        // w polu tekstowym, tak aby użytkownik mógł ją zmienić.
        // Dodatkowo, obok pola wyświetlamy przycisk 
        // "Aktualizuj zamówienie", który powoduje ponowne 
        // przesłanie tej samej strony na serwer, przy czym
        // zmieniana jest liczba zamawianych egzemplarzy 
        // danego produktu.
        for(int i=0; i<itemsOrdered.size(); i++) {
          order = (ItemOrder)itemsOrdered.get(i);
          out.println
            ("<TR>\n" +
             "  <TD>" + order.getItemID() + "</TD>\n" +
             "  <TD>" + order.getShortDescription() + "</TD>\n" +
             "  <TD>" +
             formatter.format(order.getUnitCost()) + "</TD>\n" +
             "  <TD>" +
             "<FORM>\n" +  // Przesyłamy formularz pod ten 
                           // sam adres URL
             "<INPUT TYPE=\"HIDDEN\" NAME=\"itemID\"\n" +
             "       VALUE=\"" + order.getItemID() + "\">\n" +
             "<INPUT TYPE=\"TEXT\" NAME=\"numItems\"\n" +
             "       SIZE=3 VALUE=\"" + 
             order.getNumItems() + "\">\n" +
             "<SMALL>\n" +
             "<INPUT TYPE=\"SUBMIT\"\n "+
             "       VALUE=\"Aktualizuj zamówienie\">\n" +
             "</SMALL>\n" +
             "</FORM></TD>\n" +
             "  <TD>" +
             formatter.format(order.getTotalCost()) +
             "</TR>");
        }
        String checkoutURL =
          response.encodeURL("../Checkout.html");
        // Pod tabelą wyświetlany jest przycisk "Rozliczenie"
        out.println
          ("</TABLE>\n" +
           "<FORM ACTION=\"" + checkoutURL + "\">\n" +
           "<BIG><CENTER>\n" +
           "<INPUT TYPE=\"SUBMIT\"\n" +
           "       VALUE=\"Rozliczenie\">\n" +
           "</CENTER></BIG></FORM>");
      }
      out.println("</BODY></HTML>");
    }
  }
}
