package coreservlets;

import java.util.*;
import coreservlets.beans.*;

/** Proste narzędzia to nawiązywania połączeń JDBC z 
 *  różnymi bazami danych. Sterowniki jakie będą ładowane
 *  są podane na stałe i można je dostosować do konkretnego
 *  środowiska. Można zmodyfikować kod tej klasy i ponownie
 *  ją skompilować, bądź użyć klasy 
 *  <CODE>DriverUtilities2</CODE> aby załadować sterownik
 *  na podstawie informacji podanych w pliku XML.
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall & Larry Brown; można kopiować i modyfikować bez ograniczeń.
 */

public class DriverUtilities {
  public static final String MSACCESS = "MSACCESS";
  public static final String MYSQL = "MYSQL";
  public static final String ORACLE = "ORACLE"; 

  // Tu można dodawać stałe określające typ używanej
  // bazy danych...
  
  protected static Map driverMap = new HashMap();

  /** Wczytanie informacji o sterownikach dla różnych baz
   *  danych. Metoda zawiera podane na stałe informacje
   *  o konfiugracji baz dla systemu Autorów.
   *  Czytelnik powinien dostosować podane informacje do 
   *  używanego środowiska.
   *  Można także użyć klasy <CODE>DriverUtilities2</CODE>
   *  aby wczytywać niezbędne informacje z pliku XML.
   *  <P>
   *  Każdy typ serwera bazy danych jest reprezentowany jako
   *  obiekt <CODE>DriverInfoBean</CODE>, który definiuje:
   *  typ serwera (słowo kluczowe), opis, nazwę klasy oraz 
   *  adres URL. Typ stanowi słowo kluczowe używane do 
   *  pobierania informacji.
   *  <P>
   *  Adres URL powinien zawierać symbole specjalne
   *  [$host] oraz [$dbName], które w metodzie 
   *  <CODE>makeURL</CODE> zostaną zamienione 
   *  odpowiednio na <I>nazwę komputera</I>
   *  oraz <I>nazwę bazy danych</I>.
   */

  public static void loadDrivers() {
    String vendor, description, driverClass, url;
    DriverInfoBean info = null;

    // MSAccess
    vendor = MSACCESS;
    description = "MS Access 4.0";
    driverClass = "sun.jdbc.odbc.JdbcOdbcDriver";
    url = "jdbc:odbc:[$dbName]";
    info = new DriverInfoBean(vendor, description,
                              driverClass, url);
    addDriverInfoBean(info);
    
    // MySQL
    vendor = MYSQL;
    description = "MySQL Connector/J 3.0";
    driverClass = "com.mysql.jdbc.Driver";
    url = "jdbc:mysql://[$host]:3306/[$dbName]";
    info = new DriverInfoBean(vendor, description,
                              driverClass, url);
    addDriverInfoBean(info);

    // Oracle
    vendor = ORACLE;
    description = "Oracle9i Database";
    driverClass = "oracle.jdbc.driver.OracleDriver";
    url = "jdbc:oracle:thin:@[$host]:1521:[$dbName]";
    info = new DriverInfoBean(vendor, description,
                              driverClass, url);
    addDriverInfoBean(info);

    // Tu można dodać informacje o używanej bazie danych
  }

  /** Dodanie informacji o nowym typie bazy danych 
   *  (<CODE>DriverInfoBean</CODE>) aby skojarzyć je z dostępnymi
   *  sterownikami.
   */

  public static void addDriverInfoBean(DriverInfoBean info) {
    driverMap.put(info.getVendor().toUpperCase(), info);
  }

  /** Określa czy istieją informacje o sterowniku dla podanego
   *  typu bazy danych.
   */

  public static boolean isValidVendor(String vendor) {
    DriverInfoBean info =
      (DriverInfoBean)driverMap.get(vendor.toUpperCase());
    return(info != null);
  }

  /** Wygenerowanie adresu URL w formacie wymaganym przez
   *  sterowniki bazy danych. Podczas generowania wynikowego
   *  adresu URL symbole [$host] oraz [$dbName] (pobierane 
   *  z komponentu <CODE>DriverInfoBean</CODE> dla odpowiedniego
   *  typu bazy) są zastępowane odpowiednimi argumentami 
   *  wywołania metody.
   */
   
  public static String makeURL(String host, String dbName,
                               String vendor) {
    DriverInfoBean info =
      (DriverInfoBean)driverMap.get(vendor.toUpperCase());
    if (info == null) {
      return(null);
    }
    StringBuffer url = new StringBuffer(info.getURL());
    DriverUtilities.replace(url, "[$host]", host);
    DriverUtilities.replace(url, "[$dbName]", dbName);
    return(url.toString());
  }

  /** Pobranie w pełni kwalifikowanej nazwy klasy sterownika. */

  public static String getDriver(String vendor) {
    DriverInfoBean info =
      (DriverInfoBean)driverMap.get(vendor.toUpperCase());
    if (info == null) {
      return(null);
    } else {
      return(info.getDriverClass());
    }
  }

  /** Zastąpienie fragmentu łańcucha znaków - pierwszy 
   *  odszukany, pasujący fragment (przekazany jako agument
   *  'match' jest zastępowany przez wartość przekazaną jako 
   *  argument 'value'.
   */

  private static void replace(StringBuffer buffer,
                              String match, String value) {
    int index = buffer.toString().indexOf(match);
    if (index > 0) {
      buffer.replace(index, index + match.length(), value);
    }
  }
}   