import java.net.*;
import java.io.*;
import java.util.*;

/** Prosty serwer HTTP generujący stronę WWW
 *  przedstawiającą wszystkie informacje przesłane
 *  z klienta (zazwyczaj przeglądarki) w żądaniu HTTP.
 *  Aby użyć programu należy go uruchomić w wybranym
 *  komputerze, podając numer portu na którym
 *  ma działać (jeśli nie chcesz by działał na domyślnym
 *  porcie o numerze 8088). Następnie, na tym samym lub
 *  innym komputerze, uruchom przeglądarkę WWW 
 *  i odwołaj się do adresu http://komputer:8088/jakasStrona.
 *  Wyświetlona, wynikowa strona będzie prezentować informacje
 *  przesłane przez przeglądarkę w żądaniu HTTP. W przypadku
 *  testowania serwletów lub programów CGI, należy podać
 *  adres http://komputer:8088/jakisProgram w atrybucie 
 *  ACTION formularza. Można przesyłać dane zarówno metodą 
 *  GET jak i POST; niezależnie od użytej metody 
 *  wyniki będą przedstawiać wszystkie informacje 
 *  przesłane przez przeglądarkę.
 *  <P>
 *  Przykłady z książki Java Servlet i JavaServer Pages. Wydanie II.
 *  Wydawnictwo HELION
 *  http://helion.pl/.
 *  &copy; 2003 Marty Hall & Larry Brown; można kopiować i modyfikować bez ograniczeń.
 */

public class EchoServer extends NetworkServer {
  protected int maxRequestLines = 50;
  protected String serverName = "EchoServer";

  /** Podaj numer porty jak argument wywołania programu.
   *  Jeśli numer portu nie zostanie podany, program
   *  użyje domyślnego portu o numerze 8088.
   */

  public static void main(String[] args) {
    int port = 8088;
    if (args.length > 0) {
      try {
        port = Integer.parseInt(args[0]);
      } catch(NumberFormatException nfe) {}
    }
    new EchoServer(port, 0);
  }

  public EchoServer(int port, int maxConnections) {
    super(port, maxConnections);
    listen();
  }

  /** Przesłania metodę handleConnection klasy
   *  NetworkServer. Metoda odczytuje każdy przesłany
   *  wiersz informacji, zapisuje go w tablicy łańcuchów
   *  znaków, a następnie umieszcza na stronie WWW
   *  wewnątrz elementu PRE. Tak utworzona strona WWW
   *  jest przesyłana do przeglądarki.
   */

  public void handleConnection(Socket server)
      throws IOException{
    System.out.println
        (serverName + ": odebrano połączenie z " +
         server.getInetAddress().getHostName());
    BufferedReader in = SocketUtil.getReader(server);
    PrintWriter out = SocketUtil.getWriter(server);
    String[] inputLines = new String[maxRequestLines];
    int i;
    for (i=0; i<maxRequestLines; i++) {
      inputLines[i] = in.readLine();
      if (inputLines[i] == null) // Klient zamknął połączenie.
        break;
      if (inputLines[i].length() == 0) { // Pusty wiersz.
        if (usingPost(inputLines)) {
          readPostData(inputLines, i, in);
          i = i + 2;
        }
        break;
      }
    }
    printHeader(out);
    for (int j=0; j<i; j++) {
      out.println(inputLines[j]);
    }
    printTrailer(out);
    server.close();
  }

  // Przesyła standardową odpowiedź HTTP i początek standardowej
  // strony WWW. Dla uzyskania zgodności ze wszystkimi klientami
  // wykorzystywany jest protokół HTTP 1.0.

  private void printHeader(PrintWriter out) {
    out.println
      ("HTTP/1.0 200 OK\r\n" +
       "Server: " + serverName + "\r\n" +
       "Content-Type: text/html\r\n" +
       "\r\n" +
       "<!DOCTYPE HTML PUBLIC " +
         "\"-//W3C//DTD HTML 4.0 Transitional//EN\">\n" +
       "<HTML>\n" +
       "<HEAD>\n" +
       "  <TITLE>" + serverName + " Wyniki</TITLE>\n" +
       "<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=ISO-8859-2\">" +
       "</HEAD>\n" +
       "\n" +
       "<BODY BGCOLOR=\"#FDF5E6\">\n" +
       "<H1 ALIGN=\"CENTER\">" + serverName +
         " Wyniki</H1>\n" +
       "Oto wiersz żądania oraz nagłówki żądania HTTP \n" +
       "przesłane przez Twoją przeglądarkę:\n" +
       "<PRE>");
  }

  // Generuje koniec standardowej strony WWW.
  
  private void printTrailer(PrintWriter out) {
    out.println
      ("</PRE>\n" +
       "</BODY>\n" +
       "</HTML>\n");
  }

  // Normalne żądania dotyczące stron WWW wykorzystują
  // metodę GET, a zatem ten serwer może odczytywać
  // przesyłane informacje kolejno, po jednym wierszu.
  // Jednak formularze HTML mogą także używać metody POST.
  // W takim przypadku należy określić liczbę przesyłanych 
  // bajtów, aby było wiadomo ile dodatkowych bajtów 
  // informacji należy odczytać po zakończeniu pobierania 
  // nagłówków żądania HTTP.

  private boolean usingPost(String[] inputs) {
    return(inputs[0].toUpperCase().startsWith("POST"));
  }

  private void readPostData(String[] inputs, int i,
                            BufferedReader in)
      throws IOException {
    int contentLength = contentLength(inputs);
    char[] postData = new char[contentLength];
    in.read(postData, 0, contentLength);
    inputs[++i] = new String(postData, 0, contentLength);
  }

  // Dysponując wierszem rozpoczynającym się od łańcucha znaków
  // Content-Length, metoda zwraca zapisaną w tym wierszu
  // liczbę całkowitą.
  
  private int contentLength(String[] inputs) {
    String input;
    for (int i=0; i<inputs.length; i++) {
      if (inputs[i].length() == 0)
        break;
      input = inputs[i].toUpperCase();
      if (input.startsWith("CONTENT-LENGTH"))
        return(getLength(input));
    }
    return(0);
  }

  private int getLength(String length) {
    StringTokenizer tok = new StringTokenizer(length);
    tok.nextToken();
    return(Integer.parseInt(tok.nextToken()));
  }
}           