NajlepszeOferty
================

Wykorzystamy [ten plik](https://github.com/WinVector/PDSwR2/blob/master/BestOffers/BestOffers.md) w przykładzie ukazującym mechanizm manipulowania danymi w sposób [relacyjny](https://en.wikipedia.org/wiki/Relational_database) lub na podstawie [reguł Codda](https://en.wikipedia.org/wiki/Relational_model).

Wyznaczamy sobie następujące zadanie: określenie dwie najlepsze oferty produktowe dla każdego klienta.

Gdybyśmy przechowywali [R](https://www.r-project.org) w pamięci, moglibyśmy rozwiązać ten problem za pomocą pakietu [`data.table`](https://CRAN.R-project.org/package=data.table). Jeśli jednak nasze dane znajdują się w zdalnej bazie danych i nie chcemy kopiować ich do R, możemy skorzystać z kwerend [SQL](https://en.wikipedia.org/wiki/SQL). Albo z jeszcze lepszego rozwiązania, jakim jest generator kwerend SQL - skorzystamy tu z tego rozwiązania.

Najpierw połączymy się z prostą bazą danych przechowywaną w pamięci, ale omawiane techniki powinny również działać w przypadku dużych baz danych, takich jak Postgres.

``` r
library("rquery")
```

    ## Warning: package 'rquery' was built under R version 3.5.2

``` r
# Za pomocą pakietu DBI łączy się z bazą danych.
# W tym przypadku tworzy nową bazę danych MonetDBLite, umieszczoną w pamięci.
raw_connection <- DBI::dbConnect(MonetDBLite::MonetDBLite())

# Tworzy funkcję otaczającą rquery, umożliwiającą połączenie.
dbopts <- rq_connection_tests(raw_connection)
db <- rquery_db_info(
  connection = raw_connection,
  is_dbi = TRUE,
  connection_options = dbopts)
```

Przygotujemy teraz przykładowe dane. W rzeczywistych zastosowaniach dane już znajdują się w bazie danych, dlatego wymagane jest połączenie z nią.

``` r
# Kopiuje przykładowe dane do bazy danych. 
data_handle <- rq_copy_to(
  db, 
  'oferty',
  wrapr::build_frame(
   "nazwa_uzytkownika"  , "produkt"                            , "rabat"   , "przewidywane_powinowactwo_ofert" |
     "John"             , "Gra planszowa Pandemia"             , 0.1       , 0.8596                            |
     "Nina"             , "Gra planszowa Pandemia"             , 0.2       , 0.1336                            |
     "John"             , "Laptop Dell XPS"                    , 0.1       , 0.2402                            |
     "Nina"             , "Laptop Dell XPS"                    , 0.05      , 0.3179                            |
     "John"             , "Opowiesci z roznych kieszeni Čapeka", 0.05      , 0.2439                            |
     "Nina"             , "Opowiesci z roznych kieszeni Čapeka", 0.05      , 0.06909                           |
     "John"             , "Pioro wieczne Pelikan M200"         , 0.2       , 0.6706                            |
     "Nina"             , "Pioro wieczne Pelikan M200"         , 0.1       , 0.616                             ),
  temporary = TRUE, 
  overwrite = TRUE)
```

Spójrzmy teraz na nasze dane. Dane zdalne mają często duże rozmiary, dlatego często sprawdzamy tylko ich początkowe wystąpienia (które nie muszą stanowić reprezentatywnej próby).

``` r
cat(format(data_handle))
```

    ## table("oferty"; 
    ##   nazwa_uzytkownika,
    ##   produkt,
    ##   rabat,
    ##   przewidywane_powinowactwo_ofert)

``` r
# spójrzmy na kilka pierwszych punktów danych.
execute(db, data_handle, source_limit = 6) %.>%
  knitr::kable(.)
```

| nazwa\_uzytkownika | produkt                             |  rabat   |  przewidywane\_powinowactwo\_ofert|
|:-------------------|:------------------------------------|---------:|----------------------------------:|
| John               | Gra planszowa Pandemia              |      0.10|                            0.85960|
| Nina               | Gra planszowa Pandemia              |      0.20|                            0.13360|
| John               | Laptop Dell XPS                     |      0.10|                            0.24020|
| Nina               | Laptop Dell XPS                     |      0.05|                            0.31790|
| John               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.24390|
| Nina               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.06909|

``` r
# sprawdźmy strukturę zdalną.
rquery::rstr(db, data_handle$table_name)
```

    ## table "oferty" rquery_db_info 
    ##  nrow: 8 
    ## 'data.frame':    8 obs. of  4 variables:
    ##  $ nazwa_uzytkownika              : chr  "John" "Nina" "John" "Nina" ...
    ##  $ produkt                        : chr  "Gra planszowa Pandemia" "Gra planszowa Pandemia" "Laptop Dell XPS" "Laptop Dell XPS" ...
    ##  $ rabat                          : num  0.1 0.2 0.1 0.05 0.05 0.05 0.2 0.1
    ##  $ przewidywane_powinowactwo_ofert: num  0.86 0.134 0.24 0.318 0.244 ...

``` r
# uzyskajmy zdalne podsumowanie
rquery::rsummary(db, data_handle$table_name)
```

    ##                      column index     class nrows nna nunique     min
    ## 1               nazwa_uzytkownika     1 character     8   0       2      NA
    ## 2                         produkt     2 character     8   0       4      NA
    ## 3                           rabat     3   numeric     8   0      NA 0.05000
    ## 4 przewidywane_powinowactwo_ofert     4   numeric     8   0      NA 0.06909
    ##      max      mean         sd          lexmin
    ## 1     NA        NA         NA            John
    ## 2     NA        NA         NA Laptop Dell XPS
    ## 3 0.2000 0.1062500 0.06232117            <NA>
    ## 4 0.8596 0.3938612 0.28483707            <NA>
    ##                                lexmax
    ## 1                                Nina
    ## 2 Opowiesci z roznych kieszeni Čapeka
    ## 3                                <NA>
    ## 4                                <NA>

Skoro już wiemy, że dane są faktycznie małe, spójrzmy na ich całokształt.

``` r
DBI::dbReadTable(db$connection, data_handle$table_name) %.>%
  knitr::kable(.)
```

| nazwa\_uzytkownika | produkt                             |  rabat   |  przewidywane\_powinowactwo\_ofert|
|:-------------------|:------------------------------------|---------:|----------------------------------:|
| John               | Gra planszowa Pandemia              |      0.10|                            0.85960|
| Nina               | Gra planszowa Pandemia              |      0.20|                            0.13360|
| John               | Laptop Dell XPS                     |      0.10|                            0.24020|
| Nina               | Laptop Dell XPS                     |      0.05|                            0.31790|
| John               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.24390|
| Nina               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.06909|
| John               | Pioro wieczne Pelikan M200          |      0.20|                            0.67060|
| Nina               | Pioro wieczne Pelikan M200          |      0.10|                            0.61600|

Zakładamy, że dysponujemy powyższą tabelą bazodanową, gdzie na ofertę produktu składają się trzy elementy: jej adresat, produkt, a także rabat. Dla każdego takiego tripletu wyznaczyliśmy przewidywane powinowactwo oferty, czyli modelowane prawdopodobieństwo określające, jak bardzo dana oferta może być interesująca dla klienta. Podobnie wygenerowaliśmy zróżnicowane rabaty na podstawie powinowactwa pomiędzy klientem a poszczególnymi produktami.

Rozwiążmy teraz problem wyszukania dwóch najlepszych ofert dla każdego użytkownika za pomocą operatorów relacyjnych (operatorów Codda).

Główną pracę wykona tzw. "funkcja okna": `rank()`.

``` r
data_handle %.>%
  extend(.,
         prosta_ocena = rank(),
         partitionby = "nazwa_uzytkownika",
         orderby = "przewidywane_powinowactwo_ofert",
         reverse = "przewidywane_powinowactwo_ofert") %.>%
  execute(db, .) %.>%
  knitr::kable(.)
```

| nazwa\_uzytkownika | produkt                             |  rabat   |  przewidywane\_powinowactwo\_ofert|  prosta\_ocena|
|:-------------------|:------------------------------------|---------:|----------------------------------:|--------------:|
| Nina               | Pioro wieczne Pelikan M200          |      0.10|                            0.61600|              1|
| Nina               | Laptop Dell XPS                     |      0.05|                            0.31790|              2|
| Nina               | Gra planszowa Pandemia              |      0.20|                            0.13360|              3|
| Nina               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.06909|              4|
| John               | Gra planszowa Pandemia              |      0.10|                            0.85960|              1|
| John               | Pioro wieczne Pelikan M200          |      0.20|                            0.67060|              2|
| John               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.24390|              3|
| John               | Laptop Dell XPS                     |      0.10|                            0.24020|              4|

Zwróć uwagę, że dla każdego użytkownika ocena wyznacza kolejność w kolumnie `przewidywane_powinowactwo_ofert`. Dzięki dołączeniu kolumny rangi, a następnie wybraniu wszystkich rzędów spełniających warunek `simple_rank <= 2` możemy rozwiązać nasz problem i wyznaczyć dwie najlepsze oferty dla każdego użytkownika. Wymagana jest znajomość poszczególnych operatorów relacyjnych (`extend()` dodaje kolumny, `select_rows()` wybiera rzędy, `orderby()` sortuje je; listę operatorów znajdziesz [tutaj](https://github.com/WinVector/rquery)) a także sposób ich łączenia w celu uzyskania odpowiedniego rozwiązania (jest to umiejętność, której można się wyuczyć w miarę zdobywania doświadczenia; polecamy książkę <emphasis>Praktyki mistrza SQL. Programowanie zaawansowane</emphasis> autorstwa Joe Celko (Helion, 2016), gdyż znakomicie uczy ona myślenia w kategoriach relacyjnych).

``` r
# definiujemy sekwencję operacji. 
ops <- data_handle %.>%
  # oznaczamy każdy rząd za pomocą prostej rangi dla poszczególnych użytkowników.
  extend(.,
         prosta_ocena = rank(),
         partitionby = "nazwa_uzytkownika",
         orderby = "przewidywane_powinowactwo_ofert",
         reverse = "przewidywane_powinowactwo_ofert") %.>%
  # dla każdego użytkownika wybieramy dwa rzędy o największej randze.
  select_rows(.,
              prosta_ocena <= 2) %.>%
  # sortujemy rzędy na podstawie użytkownika i rangi produktu.
  orderby(., c("nazwa_uzytkownika", "prosta_ocena"))



# umieszczamy wynik w bazie danych,
# dzięki czemu uzyskujemy nowe wystąpienie tabeli wynikowej.
result_table <- materialize(db, ops)

# kopiujemy dane z bazy danych do R i wyświetlamy je.
DBI::dbReadTable(db$connection, result_table$table_name) %.>%
  knitr::kable(.)
```

| nazwa\_uzytkownika | produkt                             |  rabat   |  przewidywane\_powinowactwo\_ofert|  prosta\_ocena|
|:-------------------|:------------------------------------|---------:|----------------------------------:|--------------:|
| John               | Gra planszowa Pandemia              |      0.10|                            0.85960|              1|
| John               | Pioro wieczne Pelikan M200          |      0.20|                            0.67060|              2|
| Nina               | Pioro wieczne Pelikan M200          |      0.10|                            0.61600|              1|
| Nina               | Laptop Dell XPS                     |      0.05|                            0.31790|              2|

Możemy nie tylko obliczać wyniki, ale także zapisywać je, wyświetlać, a nawet udostępniać w formie diagramu.

``` r
# wyświetla plan rozwiązania.
cat(format(ops))
```

    ## table("oferty"; 
    ##   nazwa_uzytkownika,
    ##   produkt,
    ##   rabat,
    ##   przewidywane_powinowactwo_ofert) %.>%
    ##  extend(.,
    ##   prosta_ocena := rank(),
    ##   p= nazwa_uzytkownika,
    ##   o= "przewidywane_powinowactwo_ofert" DESC) %.>%
    ##  select_rows(.,
    ##    simple_rank <= 2) %.>%
    ##  orderby(., nazwa_uzytkownika, prosta_ocena)

``` r
# diagram planu kwerendy
ops %.>%
  op_diagram(.) %.>% 
  DiagrammeR::grViz(.) %.>%
  DiagrammeRsvg::export_svg(.) %.>%
  write(., file="MonetDBLite_diagram.svg")
```

![](MonetDBLite_diagram.svg)

Możemy także udowodnić, że plan został rzeczywiście przesłany do bazy danych.

``` r
ops %.>% 
  to_sql(., db) %.>% 
  cat(.)
```

    ## SELECT * FROM (
    ##  SELECT * FROM (
    ##   SELECT
    ##    "nazwa_uzytkownika",
    ##    "produkt",
    ##    "rabat",
    ##    "przewidywane_powinowactwo_ofert",
    ##    rank ( ) OVER (  PARTITION BY "nazwa_uzytkownika" ORDER BY "przewidywane_powinowactwo_ofert" DESC ) AS "prosta_ocena"
    ##   FROM (
    ##    SELECT
    ##     "nazwa_uzytkownika",
    ##     "produkt",
    ##     "rabat",
    ##     "przewidywane_powinowactwo_ofert"
    ##    FROM
    ##     "oferty"
    ##    ) tsql_65637679513721707606_0000000000
    ##  ) tsql_65637679513721707606_0000000001
    ##  WHERE "prosta_ocena" <= 2
    ## ) tsql_65637679513721707606_0000000002 ORDER BY "nazwa_uzytkownika", "prosta_ocena"

Kod SQL jest długi, a sekwencje bądź układ danych są wyrażane w postaci zagnieżdżeń wewnętrznych, przez co okazują się trudne do odczytywania lub pisania przez człowieka.

Dysponujemy także dokumentacją opisującą poszczególne kolumny i tabele, oraz które kolumny są generowane.

``` r
# wygenerowane kolumny
colnames(ops)
```

    ## [1] "nazwa_uzytkownika"                "produkt"                 
    ## [3] "rabat"                            "przewidywane_powinowactwo_ofert"
    ## [5] "prosta_ocena"

``` r
# wymagane tabele i kolumny
columns_used(ops)
```
    ## $ $oferty
    ## [1] "nazwa_uzytkownika"                "produkt"                 
    ## [3] "rabat"                            "prosta_ocena"

Możemy to samo osiągnąć za pomocą pakietu [`dbplyr`](https://CRAN.R-project.org/package=dbplyr), uważamy jednak, że pakiet [`rquery`](https://CRAN.R-project.org/package=rquery) zapewnia nie dość, że pełne, to jeszcze elastyczne rozwiązanie.

Z kolei dzięki pakietowi `rqdatatable` te same operacje można przeprowadzać w pamięci R (za pomocą pakietu `data.table`!). Dzięki temu możemy przygotowywać zdalne operacje w pamięci i okazuje się, że takie rozwiązanie jest dość szybkie (szybsze od bazowego środowiska R oraz od pakietu `dplyr`) właśnie dzięki możliwości użycia pakietu `data.table` (jednak nie jest takie szybkie jak sam pakiet `data.table` z powodu występowania pewnych dodatkowych operacji).

``` r
library("rqdatatable") # rejestruje rqdatatable jako pakiet wykonawczy

local_data <- DBI::dbReadTable(db$connection, data_handle$table_name)

knitr::kable(local_data)
```

| nazwa\_uzytkownika | produkt                             |  rabat   |  przewidywane\_powinowactwo\_ofert|
|:-------------------|:------------------------------------|---------:|----------------------------------:|
| John               | Gra planszowa Pandemia              |      0.10|                            0.85960|
| Nina               | Gra planszowa Pandemia              |      0.20|                            0.13360|
| John               | Laptop Dell XPS                     |      0.10|                            0.24020|
| Nina               | Laptop Dell XPS                     |      0.05|                            0.31790|
| John               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.24390|
| Nina               | Opowiesci z roznych kieszeni Čapeka |      0.05|                            0.06909|
| John               | Pioro wieczne Pelikan M200          |      0.20|                            0.67060|
| Nina               | Pioro wieczne Pelikan M200          |      0.10|                            0.61600|

``` r
# wykorzystuje dane lokalne
local_data %.>% 
  ops %.>%
  knitr::kable(.)
```

| nazwa\_uzytkownika | produkt                             |  rabat   |  przewidywane\_powinowactwo\_ofert|  prosta\_ocena|
|:-------------------|:------------------------------------|---------:|----------------------------------:|--------------:|
| John               | Gra planszowa Pandemia              |      0.10|                            0.85960|              1|
| John               | Pioro wieczne Pelikan M200          |      0.20|                            0.67060|              2|
| Nina               | Pioro wieczne Pelikan M200          |      0.10|                            0.61600|              1|
| Nina               | Laptop Dell XPS                     |      0.05|                            0.31790|              2|


Dysponujemy także wieloma przykładami stosowania pakietu `rquery` z innymi [bazami danych](https://github.com/WinVector/rquery/tree/master/db_examples). Uchwyty danych, a także reprezentacje tabel oraz operacji w pakiecie `rquery` celowo nie zawierają odniesień do bazy danych, co znacznie ułatwia jego wielokrotne stosowanie (tak jak w naszym przykładzie).

Język `R` zawiera także pakiet [`sqldf`](https://CRAN.R-project.org/package=sqldf) umożliwiający realizowanie kodu `SQL` za pomocą obiektów `data.frames` (poprzez kopiowanie ich do bazy danych). Uzyskujemy w ten sposób wydajniejsze kwerendy relacyjne w stosunku do możliwości pakietów `dbplyr` lub `rquery` (gdyż uzyskujemy dostęp do wszelkich typów złączeń), a w przypadku skomplikowanych kwerend może okazać się szybszy w porównaniu ze standardowym `R` lub lokalnym (niebazodanowym) pakietem `dplyr`, gdyż szybszy czas realizacji kwerend może równoważyć czas potrzebny na transfer danych. Uwaga podczas stosowania pakietu `sqldf`: nie zapomnij o wprowadzeniu wiersza `options(gsubfn.engine = "R")`, gdyż w ten sposób zapobiegniesz niepotrzebnemu uruchomieniu serwera `X11`). Uwaga: w kwestii szybkości obliczeń w pamięci (i wyrazistości notacji) zwycięzcą okazuje się być pakiet `data.table`.

``` r
DBI::dbDisconnect(raw_connection)
rm(list = c("raw_connection", "db"))
```
