// Copyright © 2016 Alan A. A. Donovan & Brian W. Kernighan.
// License: https://creativecommons.org/licenses/by-nc-sa/4.0/


// Package memo zapewnia współbieżnie bezpieczną memoizację funkcji typu Func.
// Współbieżne żądania są szeregowane przez Mutex.
package memo

import "sync"

// Func jest typem funkcji, która ma być zmemoizowana.
type Func func(string) (interface{}, error)

type result struct {
	value interface{}
	err   error
}

func New(f Func) *Memo {
	return &Memo{f: f, cache: make(map[string]result)}
}

//!+

type Memo struct {
	f     Func
	mu    sync.Mutex // strzeże cache
	cache map[string]result
}

// Get jest współbieżnie bezpieczna.
func (memo *Memo) Get(key string) (value interface{}, err error) {
	memo.mu.Lock()
	res, ok := memo.cache[key]
	if !ok {
		res.value, res.err = memo.f(key)
		memo.cache[key] = res
	}
	memo.mu.Unlock()
	return res.value, res.err
}

//!-
