/*
 * Copyright 2013 Oracle and/or its affiliates.
 * All rights reserved.  You may not modify, use,
 * reproduce, or distribute this software except in
 * compliance with  the terms of the License at:
 * http://developers.sun.com/license/berkeley_license.html
 */


package com.forest.web;

import com.forest.ejb.UserBean;
import com.forest.entity.Customer;
import com.forest.entity.Person;
import com.forest.qualifiers.LoggedIn;
import com.forest.web.util.JsfUtil;
import com.forest.web.util.MD5Util;
import com.forest.web.util.AbstractPaginationHelper;
import com.forest.web.util.PageNavigation;
import java.io.Serializable;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;
import javax.enterprise.context.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.FacesConverter;
import javax.faces.model.DataModel;
import javax.faces.model.ListDataModel;
import javax.faces.model.SelectItem;
import javax.inject.Inject;
import javax.inject.Named;


@Named(value = "customerController")
@SessionScoped
public class CustomerController implements Serializable {
    private static final String BUNDLE = "/Bundle";
    private static final long serialVersionUID = 2081269066939259737L;
    private static final Logger logger = Logger.getLogger(
                CustomerController.class.getCanonicalName());
    @Inject
    @LoggedIn
    Person authenticated;
    @EJB
    private com.forest.ejb.UserBean ejbFacade;
    private AbstractPaginationHelper pagination;
    private Customer current;
    private DataModel items = null;
    private int selectedItemIndex;

    public CustomerController() {
    }

    public Customer getSelected() {
        if (current == null) {
            current = new Customer();
            selectedItemIndex = -1;
        }

        return current;
    }

    private UserBean getFacade() {
        return ejbFacade;
    }

    public AbstractPaginationHelper getPagination() {
        if (pagination == null) {
            pagination = new AbstractPaginationHelper(
                        AbstractPaginationHelper.DEFAULT_SIZE) {
                        @Override
                        public int getItemsCount() {
                            return getFacade()
                                       .count();
                        }

                        @Override
                        public DataModel createPageDataModel() {
                            return new ListDataModel(
                                    getFacade().findRange(
                                            new int[] {
                                                getPageFirstItem(),
                                                getPageFirstItem()
                                                + getPageSize()
                                            }));

                            //return new ListDataModel(getFacade().findAll());
                        }
                    };
        }

        return pagination;
    }

    public PageNavigation prepareList() {
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation prepareView() {
        current = (Customer) getItems()
                                 .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();

        return PageNavigation.VIEW;
    }

    public PageNavigation prepareCreate() {
        current = new Customer();
        selectedItemIndex = -1;

        return PageNavigation.CREATE;
    }

    private boolean isUserDuplicated(Person p) {
        return (getFacade()
                    .getUserByEmail(p.getEmail()) == null) ? false : true;
    }

    public PageNavigation create() {
        try {
            if (!isUserDuplicated(current)) {
                // password encrypt
                current.setPassword(MD5Util.generateMD5(current.getPassword()));
                getFacade()
                    .createUser(current);
                JsfUtil.addSuccessMessage(
                        ResourceBundle.getBundle(BUNDLE).getString(
                                "CustomerCreated"));
            } else {
                JsfUtil.addErrorMessage(
                        ResourceBundle.getBundle(BUNDLE).getString(
                                "DuplicatedCustomerError"));
            }

            //return prepareCreate();
            return PageNavigation.INDEX;
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "CustomerCreationError"));

            return null;
        }
    }

    public PageNavigation prepareEdit() {
        current = (Customer) getItems()
                                 .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();

        return PageNavigation.EDIT;
    }

    public PageNavigation update() {
        try {
            logger.log(
                Level.INFO,
                "Updating customer ID:{0}",
                current.getId());
            getFacade()
                .edit(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "CustomerUpdated"));

            return PageNavigation.VIEW;
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));

            return null;
        }
    }

    public PageNavigation destroy() {
        current = (Customer) getItems()
                                 .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();
        performDestroy();
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation destroyAndView() {
        performDestroy();
        recreateModel();
        updateCurrentItem();

        if (selectedItemIndex >= 0) {
            return PageNavigation.VIEW;
        } else {
            // all items were removed - go back to list
            recreateModel();

            return PageNavigation.LIST;
        }
    }

    private void performDestroy() {
        try {
            getFacade()
                .remove(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "CustomerDeleted"));
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));
        }
    }

    private void updateCurrentItem() {
        int count = getFacade()
                        .count();

        if (selectedItemIndex >= count) {
            // selected index cannot be bigger than number of items:
            selectedItemIndex = count - 1;

            // go to previous page if last page disappeared:
            if (pagination.getPageFirstItem() >= count) {
                pagination.previousPage();
            }
        }

        if (selectedItemIndex >= 0) {
            current = getFacade()
                          .findRange(
                        new int[] { selectedItemIndex, selectedItemIndex + 1 })
                          .get(0);
        }
    }

    public DataModel getItems() {
        if (items == null) {
            items = getPagination()
                        .createPageDataModel();
        }

        return items;
    }

    private void recreateModel() {
        items = null;
    }

    public PageNavigation next() {
        getPagination()
            .nextPage();
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation previous() {
        getPagination()
            .previousPage();
        recreateModel();

        return PageNavigation.LIST;
    }

    public SelectItem[] getItemsAvailableSelectMany() {
        return JsfUtil.getSelectItems(
            ejbFacade.findAll(),
            false);
    }

    public SelectItem[] getItemsAvailableSelectOne() {
        return JsfUtil.getSelectItems(
            ejbFacade.findAll(),
            true);
    }

    public void setCustomer(Customer user) {
        this.authenticated = user;
    }

    public Person getAuthenticated() {
        return authenticated;
    }

    public void setAuthenticated(Person p) {
        this.authenticated = p;
    }

    @FacesConverter(forClass = Customer.class)
    public static class CustomerControllerConverter implements Converter {
        @Override
        public Object getAsObject(
            FacesContext facesContext,
            UIComponent component,
            String value) {
            if ((value == null) || (value.length() == 0)) {
                return null;
            }

            CustomerController controller = (CustomerController) facesContext.getApplication()
                                                                             .getELResolver()
                                                                             .getValue(
                        facesContext.getELContext(),
                        null,
                        "customerController");

            return controller.ejbFacade.find(getKey(value));
        }

        java.lang.Integer getKey(String value) {
            java.lang.Integer key;
            key = Integer.valueOf(value);

            return key;
        }

        String getStringKey(java.lang.Integer value) {
            StringBuilder sb = new StringBuilder();
            sb.append(value);

            return sb.toString();
        }

        @Override
        public String getAsString(
            FacesContext facesContext,
            UIComponent component,
            Object object) {
            if (object == null) {
                return null;
            }

            if (object instanceof Customer) {
                Customer o = (Customer) object;

                return getStringKey(o.getId());
            } else {
                throw new IllegalArgumentException(
                        "object " + object + " is of type "
                        + object.getClass().getName() + "; expected type: "
                        + CustomerController.class.getName());
            }
        }
    }
}
