package com.brackeen.javagamebook.math3D;

/**
    Klasa Vector3D implementuje trjwymiarowy wektor definiowany 
    przez liczby zmiennoprzecinkowe x, y oraz z. Tak reprezentacj mona 
    traktowa jako punkt o wsprzdnych (x,y,z) lub jako wektory wiodcy 
    od punktu (0,0,0)do punktu (x,y,z).
*/
public class Vector3D implements Transformable {

    public float x;
    public float y;
    public float z;


    /**
        Tworzy nowy Vector3D z (0,0,0).
    */
    public Vector3D() {
        this(0,0,0);
    }


    /**
        Tworzy nowy Vector3D o tych samych wartociach, jakie
        ustalono w innym wektorze Vector3D.
    */
    public Vector3D(Vector3D v) {
        this(v.x, v.y, v.z);
    }


    /**
        Tworzy nowy Vector3D o podanych wartociach (x, y, z).
    */
    public Vector3D(float x, float y, float z) {
        setTo(x, y, z);
    }


    /**
        Sprawdza, czy ten Vector3D jest rwny podanemu obiektowi (Object).
        Bd rwne tylko, jeli obiekt jest rwnie wektorem typu Vector3D,
        a wsprzdne x, y oraz z obu wektorw s rwne.
    */
    public boolean equals(Object obj) {
        Vector3D v = (Vector3D)obj;
        return (v.x == x && v.y == y && v.z == z);
    }


    /**
        Sprawdza, czy dany Vector3D ma okrelone wsprzdne
        x, y oraz z.
    */
    public boolean equals(float x, float y, float z) {
        return (this.x == x && this.y == y && this.z == z);
    }


    /**
        Przypisuje wektorowi takie same wartoci, jakie ma
        inny Vector3D.
    */
    public void setTo(Vector3D v) {
        setTo(v.x, v.y, v.z);
    }


    /**
        Przypisuje temu wektorowi ustalone wartoci (x, y, z).
    */
    public void setTo(float x, float y, float z) {
        this.x = x;
        this.y = y;
        this.z = z;
    }


    /**
        Dodaje do tego wektora okrelone wartoci(x, y, z).
    */
    public void add(float x, float y, float z) {
        this.x+=x;
        this.y+=y;
        this.z+=z;
    }


    /**
        Odejmuje od tego wektora okrelone wartoci (x, y, z).
    */
    public void subtract(float x, float y, float z) {
        add(-x, -y, -z);
    }


    /**
        Dodaje do tego wektora inny, ustalony wektor.
    */
    public void add(Vector3D v) {
        add(v.x, v.y, v.z);
    }


    /**
        Odejmuje od tego wektora inny, ustalony wektor.
    */
    public void subtract(Vector3D v) {
        add(-v.x, -v.y, -v.z);
    }


    /**
        Mnoy ten wektor przez podan warto s. Nowa dugo
        wektora bdzie si rwna length()*s.
    */
    public void multiply(float s) {
       x*=s;
       y*=s;
       z*=s;
    }


    /**
        Dzieli ten wektor przez okrelon warto s. Nowa dugo
        wektora bdzie si rwna length()/s.
    */
    public void divide(float s) {
       x/=s;
       y/=s;
       z/=s;
    }


    /**
        Zwraca dugo wektora jako liczb zmiennoprzecinkow typu float.
    */
    public float length() {
        return (float)Math.sqrt(x*x + y*y + z*z);
    }


    /**
        Przeksztaca ten Vector3D na wektor jednostkowy lub
        innymi sowy na wektor o dugoci 1. Rwnowana wywoaniu 
        v.divide(v.length()).
    */
    public void normalize() {
        divide(length());
    }


    /**
        Przeksztaca ten Vector3D na reprezentacj acuchow.
    */
    public String toString() {
        return "(" + x + ", " + y + ", " + z + ")";
    }

    /**
        Obr ten wektor naokoo osi x o okrelony kt.
        Kt podawany jest w radianach. Funkcja Math.toRadians() 
        pozwala na konwersj ze stopni na radiany.
    */
    public void rotateX(float angle) {
        rotateX((float)Math.cos(angle), (float)Math.sin(angle));
    }


    /**
        Obr ten wektor naokoo osi y o okrelony kt.
        Kt podawany jest w radianach. Funkcja Math.toRadians() 
        pozwala na konwersj ze stopni na radiany.
    */
    public void rotateY(float angle) {
        rotateY((float)Math.cos(angle), (float)Math.sin(angle));
    }


    /**
        Obr ten wektor naokoo osi z o okrelony kt.
        Kt podawany jest w radianach. Funkcja Math.toRadians() 
        pozwala na konwersj ze stopni na radiany.
    */
    public void rotateZ(float angle) {
        rotateZ((float)Math.cos(angle), (float)Math.sin(angle));
    }

    /**
        Obr ten wektor dookoa osi x o okrelony kt,
        wykorzystujc ju wyliczone wartoci cosinusa i sinusa kta,
        o ktry obracamy.
    */
    public void rotateX(float cosAngle, float sinAngle) {
        float newY = y*cosAngle - z*sinAngle;
        float newZ = y*sinAngle + z*cosAngle;
        y = newY;
        z = newZ;
    }


    /**
        Obr ten wektor dookoa osi y o okrelony kt,
        wykorzystujc ju wyliczone wartoci cosinusa i sinusa kta,
        o ktry obracamy.
    */
    public void rotateY(float cosAngle, float sinAngle) {
        float newX = z*sinAngle + x*cosAngle;
        float newZ = z*cosAngle - x*sinAngle;
        x = newX;
        z = newZ;
    }


    /**
        Obr ten wektor dookoa osi x o okrelony kt,
        wykorzystujc ju wyliczone wartoci cosinusa i sinusa kta,
        o ktry obracamy.
    */
    public void rotateZ(float cosAngle, float sinAngle) {
        float newX = x*cosAngle - y*sinAngle;
        float newY = x*sinAngle + y*cosAngle;
        x = newX;
        y = newY;
    }


    /**
        Dodaje do tego wektora okrelon transformacj. Wektor 
        jest najpierw obracany, a dopiero potem wykonywana jest translacja.
    */
    public void add(Transform3D xform) {

        // obr
        addRotation(xform);

        // wykonaj translacj
        add(xform.getLocation());
    }


    /**
        Odejmuje do tego wektora okrelon transformacj. Na wektorze 
        najpierw wykonywana jest translacja, a dopiero potem jest obracany.
    */
    public void subtract(Transform3D xform) {

        // wykonaj translacj
        subtract(xform.getLocation());

        // obr
        subtractRotation(xform);
    }


    /**
        Obraca wektor o kt podany 
        w transformacji.
    */
    public void addRotation(Transform3D xform) {
        rotateX(xform.getCosAngleX(), xform.getSinAngleX());
        rotateZ(xform.getCosAngleZ(), xform.getSinAngleZ());
        rotateY(xform.getCosAngleY(), xform.getSinAngleY());
    }


    /**
        Obraca wektor o kt przeciwny do podanego 
        w transformacji.
    */
    public void subtractRotation(Transform3D xform) {
        // zauwacie, e sin(-x) == -sin(x) i cos(-x) == cos(x)
        rotateY(xform.getCosAngleY(), -xform.getSinAngleY());
        rotateZ(xform.getCosAngleZ(), -xform.getSinAngleZ());
        rotateX(xform.getCosAngleX(), -xform.getSinAngleX());
    }

    /**
        Zwraca iloczyn skalarny tego wektora i innego podanego
        wektora.
    */
    public float getDotProduct(Vector3D v) {
        return x*v.x + y*v.y + z*v.z;
    }

    /**
        Przypisuje temu wektorowi wynik iloczynu wektorowego  dwch
        podanych wektorw. Jeden z podanych wektorw moe by 
        tym wektorem (this).
    */
    public void setToCrossProduct(Vector3D u, Vector3D v) {
        // przypisz najpierw do zmiennych lokalnych, bo u lub v mog by wektorem 'this'
        float x = u.y * v.z - u.z * v.y;
        float y = u.z * v.x - u.x * v.z;
        float z = u.x * v.y - u.y * v.x;
        this.x = x;
        this.y = y;
        this.z = z;
    }


    /**
        Zwraca podniesion do kwadratu odlego pomidzy tym wektorem
        a wektorem przekazanym w postaci argumentu.
    */
    public float getDistanceSq(Vector3D v) {
        float dx = v.x - x;
        float dy = v.y - y;
        float dz = v.z - z;
        return dx*dx + dy*dy + dz*dz;
    }


    /**
        Zwraca odlego pomidzy tym wektorem a wektorem przekazanym
        w postaci argumentu.
    */
    public float getDistance(Vector3D v) {
        return (float)Math.sqrt(getDistanceSq(v));
    }


    /**
        Ustawia dugo tego wektora.
    */
    public void setLength(float newLength) {
        normalize();
        multiply(newLength);
    }
}
