package com.brackeen.javagamebook.scripting;

import java.io.IOException;
import java.util.*;
import com.brackeen.javagamebook.game.*;
import bsh.*;

public class ScriptManager {

    /**
        GameObjectEventListener delegujcy wywoania do metod skryptowych
        ScriptedListener jest dodawany do kadego obiektu GameObject
        ktry ma co najmniej jedn metod skryptow.
    */
    public class ScriptedListener
        implements GameObjectEventListener
    {

        public void notifyVisible(GameObject object,
            boolean visible)
        {
            invoke(object.getName() +
                (visible?"Visible":"NotVisible"));
        }

        public void notifyObjectCollision(GameObject object,
            GameObject otherObject)
        {
            if (otherObject.getName() != null) {
                invoke(object.getName() + "_" +
                    otherObject.getName() + "Collision");
            }

        }

        public void notifyObjectTouch(GameObject object,
            GameObject otherObject)
        {
            if (otherObject.getName() != null) {
                invoke(object.getName() + "_" +
                    otherObject.getName() + "Touch");
            }

        }

        public void notifyObjectRelease(GameObject object,
            GameObject otherObject)
        {
            if (otherObject.getName() != null) {
                invoke(object.getName() + "_" +
                    otherObject.getName() + "Release");
            }

        }

        public void notifyFloorCollision(GameObject object) {
            invoke(object.getName() + "FloorCollision");
        }

        public void notifyCeilingCollision(GameObject object) {
            invoke(object.getName() +"CeilingCollision");
        }

        public void notifyWallCollision(GameObject object) {
            invoke(object.getName() + "WallCollision");
        }
    }

    private static final Class[] NO_ARGS = new Class[0];

    private Interpreter bsh;
    private GameObjectEventListener scriptedListener;

    public ScriptManager() {
         scriptedListener = new ScriptedListener();
    }


    /**
        Konfiguruje ScriptManager dla poziomu. Lista plikw
        skryptw do uruchomienia oraz wszystkie obiekty w
        GameObjectManager majce nazwy s dodawane jako zmienne
        nazwane dla skryptw.
        Dodatkowo wywoywana jest metoda initLevel() ze skryptu
        o ile zostaa zdefiniowana.
    */
    public void setupLevel(GameObjectManager gameObjectManager,
        GameTaskManager gameTaskManager, String[] scriptFiles)
    {
        bsh = new Interpreter();
        try {
            // wykonanie pliku rdowego (i zaadowanych metod).
            for (int i=0; i<scriptFiles.length; i++) {
                bsh.source(scriptFiles[i]);
            }

            bsh.set("gameTaskManager", gameTaskManager);

            // Wszystkie nazwane obiekty s traktowane 
            // jako zmienne w beanshell.
            // UWAGA: powoduje to utrzymywanie wszystkich obiektw
            // w pamici a do nastpnego poziomu nawet, jeeli 
            // zostay w grze zniszczone.
            Iterator i = gameObjectManager.iterator();
            while (i.hasNext()) {
                GameObject object = (GameObject)i.next();
                if (object.getName() != null) {
                    bsh.set(object.getName(), object);

                    // dodanie nasuchu skryptowego dla obiektu
                    if (hasScripts(object)) {
                        object.addListener(scriptedListener);
                    }
                }
            }

            // inicjowanie kodu poziomu - wywoanie initLevel()
            invoke("initLevel");
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }
        catch (EvalError ex) {
            ex.printStackTrace();
        }
    }

    /**
        Sprawdza, czy okrelony obiekt gry posiada skrypt
        (czy metoda skryptu zaczyna si od nazwy obiektu).
    */
    public boolean hasScripts(GameObject object) {
        if (object.getName() != null) {
            String[] names = bsh.getNameSpace().getMethodNames();
            for (int i=0; i<names.length; i++) {
                if (names[i].startsWith(object.getName())) {
                    return true;
                }
            }
        }
        // nic nie znaleziono
        return false;
    }

    /**
        Zwraca true jeeli podana nazwa metody jest istniejc
        metod skryptow.
    */
    public boolean isMethod(String methodName) {
        return (bsh.getNameSpace().
            getMethod(methodName, NO_ARGS) != null);
    }

    /**
        Wywouje okrelona metod skryptow.
    */
    public void invoke(String methodName) {
        if (isMethod(methodName)) {
            try {
                bsh.eval(methodName + "()");
            }
            catch (EvalError e) {
                e.printStackTrace();
            }
        }
    }

}
