package com.brackeen.javagamebook.game;

import com.brackeen.javagamebook.math3D.*;

/**
    GameObject jest klas bazow dla obiektw dowolnego typu,
    ktre wystpuj w grze i s przechowywane w obiekcie klasy
    PolygonGroup. Przykadowo, obiekt klasy GameObject moe
    reprezentowa obiekt statyczny (np. skrzyni), obiekt w
    ruchu (np. pocisk lub przeciwnika) lub dowolny inny typ
    obiektu (np. apteczk). Obiekty klasy GameObject maj trzy
    podstawowe stany: STATE_IDLE, STATE_ACTIVE lub STATE_DESTROYED.
*/
public class GameObject {

    /**
        Reprezentuje bezczynny obiekt GameObject. Jeli obiekt jest
        bezczynny, jego transformacja Transform3D nie jest aktualizowana. 
        Domylnie GameObjects s pocztkowo bezczynne i staj si aktywne
        dopiero, gdy s po raz pierwszy obserwowane. Mona to zmieni
        przesaniajc metod notifyVisible().
    */
    protected static final int STATE_IDLE = 0;


    /**
        Reprezentuje aktywny obiekt GameObject.
        Obiekt taki bdzie aktualizowany i rysowany.
        tj. jego transformacja Transform3D moe by 
        aktualizowana.
    */
    protected static final int STATE_ACTIVE = 1;

    /**
        Reprezentuje obiekt GameObject, ktry zosta zniszczony 
        i nie powinien by ju wicej aktualizowany lub rysowany.
        W stanie STATE_DESTROYED, manager GameObjectManager
        powinien usun ten obiekt GameObject z listy obiktw
        gry, ktr zarzdza.
    */
    protected static final int STATE_DESTROYED = 2;

    private PolygonGroup polygonGroup;
    private PolygonGroupBounds bounds;
    private int state;
    private boolean isJumping;
    private float floorHeight;
    private float ceilHeight;

    /**
        Tworzy nowy obiekt klasy GameObject reprezentowany przez
        wskazany obiekt klasy PolygonGroup. Obiekt PolygonGroup
        moe by rwny null.
    */
    public GameObject(PolygonGroup polygonGroup) {
        this.polygonGroup = polygonGroup;
        bounds = new PolygonGroupBounds(polygonGroup);
        state = STATE_IDLE;
    }


    /**
        Zwraca pooenie tego obiektu GameObject w oparciu o
        metod klasy Transform3D.
    */
    public Vector3D getLocation() {
        return polygonGroup.getTransform().getLocation();
    }


    /**
        Zwraca informacje o ruchu obiektu.
    */
    public MovingTransform3D getTransform() {
        return polygonGroup.getTransform();
    }


    /**
        Zwraca obiekt PolygonGroup, do ktrego naley ten obiekt.
    */
    public PolygonGroup getPolygonGroup() {
        return polygonGroup;
    }


    /**
        Zwraca ograniczenia obiektu PolygonGroup.
    */
    public PolygonGroupBounds getBounds() {
        return bounds;
    }


    /**
        Zwraca wsprzdn X tego obiektu GameObject.
    */
    public float getX() {
        return getLocation().x;
    }


    /**
        Zwraca wsprzdn Y tego obiektu GameObject.
    */
    public float getY() {
        return getLocation().y;
    }


    /**
        Zwraca wsprzdn Z tego obiektu GameObject.
    */
    public float getZ() {
        return getLocation().z;
    }


    /**
        Rejestruje wysoko podogi pod tym obiektem GameObject.
    */
    public void setFloorHeight(float floorHeight) {
        this.floorHeight = floorHeight;
    }


    /**
        Rejestruje wysoko sufitu nad tym obiektem GameObject.
    */
    public void setCeilHeight(float ceilHeight) {
        this.ceilHeight = ceilHeight;
    }


    /**
        Zwraca wysoko podogi ustawion za pomoc metody setFloorHeight.
    */
    public float getFloorHeight() {
        return floorHeight;
    }


    /**
        Zwraca wysoko sufitu ustawion za pomoc metody setCeilHeight.
    */
    public float getCeilHeight() {
        return ceilHeight;
    }


    /**
        Ustawia stan tego obiektu. Dostpne wartoci to:
        STATE_IDLE, STATE_ACTIVE lub STATE_DESTROYED.
    */
    protected void setState(int state) {
        this.state = state;
    }


    /**
        Ustawia stan wskazanego obiektu. W ten sposb dowolny obiekt
        GameObject moe ustawia stan dowolnego innego obiektu GameObject.
        Dostpne wartoci to: STATE_IDLE, STATE_ACTIVE lub STATE_DESTROYED.
    */
    protected void setState(GameObject object, int state) {
        object.setState(state);
    }


    /**
        Sprawdza, czy ten obiekt GameObject aktualnie leci. Lecce
        obiekty nie powinny podlega wpywowi siy grawitacji.
        Domylnie zwraca false.
    */
    public boolean isFlying() {
        return false;
    }


    /**
        Sprawdza, czy dla tego obiektu GameObject jest ustawiona flaga skakania.
        GameObjectManager moe inaczej traktowa skaczce obiekty.
    */
    public boolean isJumping() {
        return isJumping;
    }


    /**
        Ustawia flag skakania tego obiektu GameObject.
        GameObjectManager moe inaczej traktowa skaczce obiekty.
    */
    public void setJumping(boolean b) {
        isJumping = b;
    }


    /**
        Zwraca true, jeli ten obiekt GameObject jest w stanie bezczynnoci.
    */
    public boolean isIdle() {
        return (state == STATE_IDLE);
    }


    /**
        Zwraca true, jeli ten obiekt GameObject jest aktywny.
    */
    public boolean isActive() {
        return (state == STATE_ACTIVE);
    }


    /**
        Zwraca true, jeli ten obiekt GameObject jest zniszczony.
    */
    public boolean isDestroyed() {
        return (state == STATE_DESTROYED);
    }


    /**
        Jeli ten GameObject jest w stanie aktywnoci, metoda
        aktualizuje jego obiekt PolygonGroup. W przeciwnym przypadku,
        metoda nie podejmuje adnych dziaa.
    */
    public void update(GameObject player, long elapsedTime) {
        if (isActive()) {
            polygonGroup.update(elapsedTime);
        }
    }


    /**
        Informuje ten GameObject o tym, czy jest widoczny po ostatniej
        aktualizacji. Jeli GameObject jest w stanie bezczynnoci i otrzymuje
        informacj, e sta si widoczyn, domylnie przechodzi w stan
        aktywnoci.
    */
    public void notifyVisible(boolean visible) {
        if (visible && isIdle()) {
            state = STATE_ACTIVE;
        }
    }


    /**
        Informuje ten GameObject, e podczas ruchu wystpia kolizja ze
        wskazanym obiektem. Metoda domylnie nie podejmuje adnych dziaa.
    */
    public void notifyObjectCollision(GameObject otherObject) {
        // nic nie rb
    }


    /**
        Informuje ten GameObject, e podczas ruchu wystpia kolizja z
        podog. Metoda domylnie nie podejmuje adnych dziaa.
    */
    public void notifyFloorCollision() {
        // nic nie rb
    }


    /**
        Informuje ten GameObject, e podczas ruchu wystpia kolizja z
        sufitem. Metoda domylnie nie podejmuje adnych dziaa.
    */
    public void notifyCeilingCollision() {
        // nic nie rb
    }


    /**
        Informuje ten GameObject, e podczas ruchu wystpia kolizja ze
        cian. Metoda domylnie nie podejmuje adnych dziaa.
    */
    public void notifyWallCollision() {
        // nic nie rb
    }

}
