package com.brackeen.javagamebook.sound;

import java.io.FilterInputStream;
import java.io.InputStream;
import java.io.IOException;

/**
    Klasa FilteredSoundStream dziedziczy po FilterInputStream i przesya
    wejciowy strumie dwikowy do SoundFilter
    @see SoundFilter
*/
public class FilteredSoundStream extends FilterInputStream {

    private static final int REMAINING_SIZE_UNKNOWN = -1;

    private SoundFilter soundFilter;
    private int remainingSize;

    /**
        Tworzy nowy obiekt FilteredSoundStream z podanymi obiektami
        InputStream oraz SoundFilter.
    */
    public FilteredSoundStream(InputStream in,
        SoundFilter soundFilter)
    {
        super(in);
        this.soundFilter = soundFilter;
        remainingSize = REMAINING_SIZE_UNKNOWN;
    }

    /**
        Przesania metod z FilterInputStream w celu zastosowania filtra
        w czasie odczytu danych.
    */
    public int read(byte[] samples, int offset, int length)
        throws IOException
    {
        // odczytuje i filtruje prbki dwikowe w strumieniu
        int bytesRead = super.read(samples, offset, length);
        if (bytesRead > 0) {
            soundFilter.filter(samples, offset, bytesRead);
            return bytesRead;
        }

        // jeeli nie ma ju bajtw w strumieniu dwikowym,
        // sprawdza, czy filtr nie ma dodatkowych bajtw ("echo").
        if (remainingSize == REMAINING_SIZE_UNKNOWN) {
            remainingSize = soundFilter.getRemainingSize();
            // zaokrglenie w d do najbliszej wielokrotnoci 4
            // (typowa wielko ramki)
            remainingSize = remainingSize / 4 * 4;
        }
        if (remainingSize > 0) {
            length = Math.min(length, remainingSize);

            // czyszczenie bufora
            for (int i=offset; i<offset+length; i++) {
                samples[i] = 0;
            }

            // filtrowanie pozostaych bajtw
            soundFilter.filter(samples, offset, length);
            remainingSize-=length;

            // koniec
            return length;
        }
        else {
            // koniec strumienia
            return -1;
        }
    }
}
