package com.hypefiend.javagamebook.client;

import com.hypefiend.javagamebook.common.*;

import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import java.net.*;
import java.io.*;

import org.apache.log4j.*;

/**
 * GameClient.java
 *
 *
 * @author <a href="mailto:bret@hypefiend.com">bret barker</a>
 * @version 1.0
 */
public abstract class GameClient extends Thread{
    /** obiekt rejestracji log4j */
    protected static Logger log = Logger.getLogger("GameClient");

    /** adres of serwera */
    protected InetAddress serverAddress;
    /** poczenie z  serwerem */
    protected SocketChannel channel;

    /** kolejka na przychodzce zdarzenia */
    protected EventQueue inQueue;
    /** kolejka na zdarzenia wychodzce */
    protected EventQueue outQueue;
    
    /** referencja do NIOEventReader, ktra odczytuje zdarzenia z serwera */
    protected NIOEventReader netReader;

    /** bufor na na zdarzenia wychodzce*/
    protected ByteBuffer writeBuffer;

    /** identyfikator naszego gracza */
    protected String playerId;
    /** identyfikator biecego przeciwnika*/
    protected String opponentId;

    /** czy obecnie gramy? */
    protected boolean inGame = false;

    /** nadal dziaa? */
    protected boolean running = true;

    /** 
     * inicjalizacja
     */
    public void init(String args[]) {    
        inQueue = new EventQueue("GameClient-in");
        outQueue = new EventQueue("GameClient-out");
        writeBuffer = ByteBuffer.allocate(Globals.MAX_EVENT_SIZE );

        try {
            serverAddress = InetAddress.getByName(args[0]);
        }
        catch (UnknownHostException uhe) {
            log.error("nieznany host: " + args[0]);
            System.exit(1);
        }
        this.playerId = args[1];

        // podczenie do serwera
        if (!connect()) 
            System.exit(1);
        
        // uruchomienie czytania z sieci
        netReader = new NIOEventReader(this, channel, inQueue);
        netReader.start();
    
    }

    public void run() {
        // logowanie
        login();

        // oczeliwanie na LOGIN_ACK
        threadSleep(200L);

        // gwna ptla
        while(running) {
            processIncomingEvents();
            writeOutgoingEvents();

            // chwila przerwy, ptla nie musi dziaa zbyt szybko
            // gdy nic si nie dzieje
            threadSleep(50);
        }
    }


    /** 
     * klasy pochodne musza implementowa metod zwracajc nazw gry
     */
    public abstract String getGameName();

    /**
     * klasy pochodne musz implementowa t metod factory
     */
    public abstract GameEvent createGameEvent();

    /**
     * oraz t, ktra tworzy zdarzenie logowania
     */
    public abstract GameEvent createLoginEvent();
    
    /**
     * oraz zdarzenie rozczenia
     */
    public abstract GameEvent createDisconnectEvent(String reason);
    
    /**
     * obsuga przychodzcych GameEvents z EventQueue
     */
    protected abstract void processIncomingEvents();

    /** 
     * zapis wszystkich zdarze oczekujcych w kolejce wychodzcej
     */
    private void writeOutgoingEvents() {
        GameEvent outEvent;
        while (outQueue.size() > 0) {
            try {
            outEvent = outQueue.deQueue();
            writeEvent(outEvent);
            }
            catch (InterruptedException ie) {}
        }    
    }
    
    /**
     * podczenie do serwera
     */
    protected boolean connect() {
        log.info("connect()");
        try {
            // otwarcie kanau gniazda
            channel = SocketChannel.open(new InetSocketAddress(serverAddress, Globals.PORT));
            channel.configureBlocking(false);

             // nie korzystamy z algorytmu Nagla
            channel.socket().setTcpNoDelay(true);
            return true;
        }
        catch (ConnectException ce) {
            log.error("Wyjtek przy connect: " + ce.getMessage());
            return false;
        }
        catch (Exception e) {
            log.error("Wyjtek przy podczaniu", e);
            return false;
        }
    }

    /**
     * wysanie zdarzenia logowania
     */
    protected void login() {
        GameEvent e = createLoginEvent();
        e.setGameName(getGameName());
        e.setPlayerId(playerId);
        writeEvent(e);
    }

    /**
     * wyczenie klienta
     * zatrzymanie czytania i zamknicie kanau
     */
    protected void shutdown() {
        running = false;
        netReader.shutdown();
        //    consoleReader.shutdown();
        try {
            channel.close();
        }
        catch (IOException ioe) {
            log.error("wyjtek przy zamykaniu kanau", ioe);
        }
    }

    /** 
     * wysyanie zdarzenia do serwera
     */
    protected void writeEvent(GameEvent ge) {
        // ustawienie nazwy gry i identyfikatora gracza
        ge.setGameName(getGameName());
        ge.setPlayerId(playerId);

        NIOUtils.prepBuffer(ge, writeBuffer);
        NIOUtils.channelWrite(channel, writeBuffer);
    }

    /** 
     * metoda uytkowa wywoujca Thread.sleep()
     */
    private void threadSleep(long time) {
        try { 
            Thread.sleep(time); 
        } 
        catch(InterruptedException e) {}
    }

    public void stdOut(String str) {
        if ((str != null) && !str.equals(""))
            System.out.println("\n" + str);
        if (inGame)
            System.out.print( playerId + " kontra " + opponentId + " > " );
        else
            System.out.print( playerId + " > " );
    }
    
    public void stdErr(String str) {
        System.err.println("\n" + str);
        }
        
        public void setOpponent(String opp) {
        opponentId = opp;
        }
}    

