package com.brackeen.javagamebook.sound;

import java.io.*;
import javax.sound.sampled.*;
import javax.sound.midi.*;
import com.brackeen.javagamebook.util.ThreadPool;
import com.brackeen.javagamebook.util.LoopingByteInputStream;


/**
    Klasa SoundManager zarzdza odtwarzaniem dwiku. Dziedziczy ona po
    klasie ThreadPool - kady z wtkw odtwarza w danym momencie jeden dwik.
    Pozwala to na atwe ograniczenie jednoczenie odtwarzanych dwikw.
    <p>Moliwe rozszerzenia tej klasy:<ul>
    <li>dodanie metody setMasterVolume(), wykorzystujcej Controls do
        ustawienia gonoci we wszystkich obiektach Line;
    <li>blokowanie odtwarzania dwiku, jeeli od dania mino wicej
        ni na przykad 500 ms.
    </ul>
*/
public class SoundManager extends ThreadPool {

    private AudioFormat playbackFormat;
    private ThreadLocal localLine;
    private ThreadLocal localBuffer;
    private Object pausedLock;
    private boolean paused;

    /**
        Tworzy nowy obiekt SoundManager, pozwalajcy na jednoczesne
        odtwarzanie podanej liczby dwikw.
    */
    public SoundManager(AudioFormat playbackFormat) {
        this(playbackFormat,
            getMaxSimultaneousSounds(playbackFormat));
    }

    /**
        Tworzy nowy obiekt SoundManager, pozwalajcy na jednoczesne
        odtwarzanie podanej liczby dwikw.
    */
    public SoundManager(AudioFormat playbackFormat,
        int maxSimultaneousSounds)
    {
        super(Math.min(maxSimultaneousSounds,
            getMaxSimultaneousSounds(playbackFormat)));
        this.playbackFormat = playbackFormat;
        localLine = new ThreadLocal();
        localBuffer = new ThreadLocal();
        pausedLock = new Object();
        // powiedomienie wtkw w puli
        synchronized (this) {
            notifyAll();
        }
    }

    /**
        Zwraca maksymaln liczb jednoczenie odtwarzanych dwikw
        w formacie okrelonym przez AudioFormat.
    */
    public static int getMaxSimultaneousSounds(
        AudioFormat playbackFormat)
    {
        DataLine.Info lineInfo = new DataLine.Info(
            SourceDataLine.class, playbackFormat);
        Mixer mixer = AudioSystem.getMixer(null);
        return mixer.getMaxLines(lineInfo);
    }


    /**
        Wykonuje czyszczenie przed zamkniciem.
    */
    protected void cleanUp() {
        // Wyczenie pauzy:
        setPaused(false);

        // Wybr miksera (zatrzymuje wszystkie dwiki):
        Mixer mixer = AudioSystem.getMixer(null);
        if (mixer.isOpen()) {
            mixer.close();
        }
    }

    public void close() {
        cleanUp();
        super.close();
    }


    public void join() {
        cleanUp();
        super.join();
    }

    /**
        Wcza tryb pauzy. Dwiki mog nie zosta przerwane natychmiast.
    */
    public void setPaused(boolean paused) {
        if (this.paused != paused) {
            synchronized (pausedLock) {
                this.paused = paused;
                if (!paused) {
                    // Przywrcenie dwikw:
                    pausedLock.notifyAll();
                }
            }
        }
    }

    /**
        Zwraca stan pauzy.
    */
    public boolean isPaused() {
        return paused;
    }

    /**
        aduje dwik z systemu plikw. W przypadku bdu zwraca warto null.
    */
    public Sound getSound(String filename) {
        return getSound(getAudioInputStream(filename));
    }

    /**
        aduje dwik ze strumienia dwikowego.
        W przypadku bdu zwraca warto null.
    */
    public Sound getSound(InputStream is) {
        return getSound(getAudioInputStream(is));
    }

    /**
        aduje dwik ze strumienia dwikowego.
    */
    public Sound getSound(AudioInputStream audioStream) {
        if (audioStream == null) {
            return null;
        }

        // Liczba bajtw do odczytania:
        int length = (int)(audioStream.getFrameLength() *
            audioStream.getFormat().getFrameSize());

        // Odczyt caego strumienia:
        byte[] samples = new byte[length];
        DataInputStream is = new DataInputStream(audioStream);
        try {
            is.readFully(samples);
            is.close();
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }

        // Zwracanie prbek:
        return new Sound(samples);
    }

    /**
        Tworzy obiekt AudioInputStream z pliku dwikowego
        zapisanego w systemie plikw.
    */
    public AudioInputStream getAudioInputStream(String filename) {
        try {
            return getAudioInputStream(
                new FileInputStream(filename));
        }
        catch (IOException ex) {
            ex.printStackTrace();
            return null;
        }
    }

    /**
        Tworzy obiekt AudioInputStream z pliku dwikowego
        przesanego za pomoc strumienia.
    */
    public AudioInputStream getAudioInputStream(InputStream is) {

        try {
            if (!is.markSupported()) {
                is = new BufferedInputStream(is);
            }
            // Otwarcie strumienia rdowego:
            AudioInputStream source =
                AudioSystem.getAudioInputStream(is);

            // Konwersja na format odtwarzania:
            return AudioSystem.getAudioInputStream(
                playbackFormat, source);
        }
        catch (UnsupportedAudioFileException ex) {
            ex.printStackTrace();
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }
        catch (IllegalArgumentException ex) {
            ex.printStackTrace();
        }

        return null;
    }

    /**
        Odtwarzanie dwiku. Metoda natychmiast koczy dziaanie.
    */
    public InputStream play(Sound sound) {
        return play(sound, null, false);
    }

    /**
        Odtwarzanie dwiku z opcjonalnym filtrem SoundFilter oraz
        opcjonalnym zaptleniem. Metoda natychmiast koczy dziaanie.
    */
    public InputStream play(Sound sound, SoundFilter filter,
        boolean loop)
    {
        InputStream is;
        if (sound != null) {
            if (loop) {
                is = new LoopingByteInputStream(
                    sound.getSamples());
            }
            else {
                is = new ByteArrayInputStream(sound.getSamples());
            }

            return play(is, filter);
        }
        return null;
    }

    /**
        Odtwarzanie dwiku ze strumienia InputStream. 
        Metoda natychmiast koczy dziaanie.
    */
    public InputStream play(InputStream is) {
        return play(is, null);
    }

    /**
        Odtwarzanie dwiku ze strumienia InputStream z 
        opcjonalnym filtrem SoundFilter.
        Metoda natychmiast koczy dziaanie.
    */
    public InputStream play(InputStream is, SoundFilter filter) {
        if (is != null) {
            if (filter != null) {
                is = new FilteredSoundStream(is, filter);
            }
            runTask(new SoundPlayer(is));
        }
        return is;
    }

    /**
        Sygnalizuje moment uruchomienia PooledThread. 
        Tworzy obiekt Line oraz bufor.
    */
    protected void threadStarted() {
        // Oczekiwanie na zakoczenie konstruktora SoundManager:
        synchronized (this) {
            try {
                wait();
            }
            catch (InterruptedException ex) { }
        }

        // Uywamy maego bufora - 100ms (1/10 s) -
        // dla filtrw zmieniajcych si w czasie dziaania.
        int bufferSize = playbackFormat.getFrameSize() *
            Math.round(playbackFormat.getSampleRate() / 10);

        // Utworzenie, otwarcie i uruchomienie obiektu Line:
        SourceDataLine line;
        DataLine.Info lineInfo = new DataLine.Info(
            SourceDataLine.class, playbackFormat);
        try {
            line = (SourceDataLine)AudioSystem.getLine(lineInfo);
            line.open(playbackFormat, bufferSize);
        }
        catch (LineUnavailableException ex) {
            // Obiekt Line jest niedostpny - sygnalizacja zakoczenia wtku:
            Thread.currentThread().interrupt();
            return;
        }

        line.start();

        // Tworzenie bufora:
        byte[] buffer = new byte[bufferSize];

        // Ustawienie zmiennych lokalnych wtku:
        localLine.set(line);
        localBuffer.set(buffer);
    }

    /**
        Sygnalizacja zatrzymania PooledThread. Oprnia i zatrzymuje
        obiekt Line dla wtku.
    */
    protected void threadStopped() {
        SourceDataLine line = (SourceDataLine)localLine.get();
        if (line != null) {
            line.drain();
            line.close();
        }
    }

    /**
        Klasa SoundPlayer jest zadaniem wykonywanym przez wtki PooledThreads.
        Otrzymuje obiekt Line z wtku oraz bufor ze zmiennych
        ThreadLocal i odtwarza dwik z InputStream.
        <p>Klasa ta dziaa tylko wtedy, gdy jest wywoana z PooledThread.
    */
    protected class SoundPlayer implements Runnable {

        private InputStream source;

        public SoundPlayer(InputStream source) {
            this.source = source;
        }

        public void run() {
            // Pobranie Line i bufora z ThreadLocals:
            SourceDataLine line = (SourceDataLine)localLine.get();
            byte[] buffer = (byte[])localBuffer.get();
            if (line == null || buffer == null) {
                // Obiekt Line niedostpny.
                return;
            }

            // Kopiowanie danych do Line:
            try {
                int numBytesRead = 0;
                while (numBytesRead != -1) {
                    // Jeeli wczona jest pauza, wycz j.
                    synchronized (pausedLock) {
                        if (paused) {
                            try {
                                pausedLock.wait();
                            }
                            catch (InterruptedException ex) {
                                return;
                            }
                        }
                    }
                    // Kopiowanie danych:
                    numBytesRead =
                        source.read(buffer, 0, buffer.length);
                    if (numBytesRead != -1) {
                        line.write(buffer, 0, numBytesRead);
                    }
                }
            }
            catch (IOException ex) {
                ex.printStackTrace();
            }
        }
    }
}
