package com.brackeen.javagamebook.math3D;

/**
    Klasa Polygon3D przedstawia wielokt jako zbir
    wierzchokw.
*/
public class Polygon3D implements Transformable {

    // tymczasowe wektory wykorzystywane do oblicze
    private static Vector3D temp1 = new Vector3D();
    private static Vector3D temp2 = new Vector3D();

    private Vector3D[] v;
    private int numVertices;
    private Vector3D normal;

    /**
        Tworzy pusty wielokt, ktry mona wykorzystywa jako
        wielokt roboczy dla potrzeb transformacji, rzutowania, itd. 
    */
    public Polygon3D() {
        numVertices = 0;
        v = new Vector3D[0];
        normal = new Vector3D();
    }


    /**
       Tworzy nowy wielokt Polygon3D o podanych wierzchokach.
    */
    public Polygon3D(Vector3D v0, Vector3D v1, Vector3D v2) {
        this(new Vector3D[] { v0, v1, v2 });
    }


    /**
        Tworzy nowy wielokt Polygon3D o podanych wierzchokach. Zakadamy,
        e wszystkie wierzchoki pooone s w tej samej paszczynie.
    */
    public Polygon3D(Vector3D v0, Vector3D v1, Vector3D v2,
        Vector3D v3)
    {
        this(new Vector3D[] { v0, v1, v2, v3 });
    }


    /**
        Tworzy nowy wielokt Polygon3D o podanych wierzchokach. Zakadamy,
        e wszystkie wierzchoki pooone s w tej samej paszczynie.
    */
    public Polygon3D(Vector3D[] vertices) {
        this.v = vertices;
        numVertices = vertices.length;
        calcNormal();
    }


    /**
        Przypisuje temu wieloktowi takie same wierzchoki jakie ma 
        podany wielokt.
    */
    public void setTo(Polygon3D polygon) {
        numVertices = polygon.numVertices;
        normal.setTo(polygon.normal);

        ensureCapacity(numVertices);
        for (int i=0; i<numVertices; i++) {
            v[i].setTo(polygon.v[i]);
        }
    }


    /**
        Upewnia si, e wielokt ma wystarczajc pojemno,
        eby przechowywa dane o wszystkich swoich wierzchokach.
    */
    protected void ensureCapacity(int length) {
        if (v.length < length) {
            Vector3D[] newV = new Vector3D[length];
            System.arraycopy(v,0,newV,0,v.length);
            for (int i=v.length; i<newV.length; i++) {
                newV[i] = new Vector3D();
            }
            v = newV;
        }
    }


    /**
        Pobiera liczb wierzchokw, ktre ma ten wielokt.
    */
    public int getNumVertices() {
        return numVertices;
    }


    /**
        Pobiera wierzchoek o podanym indeksie.
    */
    public Vector3D getVertex(int index) {
        return v[index];
    }


    /**
        Rzutuje ten wielokt na okno obrazu.
    */
    public void project(ViewWindow view) {
        for (int i=0; i<numVertices; i++) {
            view.project(v[i]);
        }
    }


    // methods from the Transformable interface.

    public void add(Vector3D u) {
       for (int i=0; i<numVertices; i++) {
           v[i].add(u);
       }
    }

    public void subtract(Vector3D u) {
       for (int i=0; i<numVertices; i++) {
           v[i].subtract(u);
       }
    }

    public void add(Transform3D xform) {
        addRotation(xform);
        add(xform.getLocation());
    }

    public void subtract(Transform3D xform) {
        subtract(xform.getLocation());
        subtractRotation(xform);
    }

    public void addRotation(Transform3D xform) {
        for (int i=0; i<numVertices; i++) {
           v[i].addRotation(xform);
        }
        normal.addRotation(xform);
    }

    public void subtractRotation(Transform3D xform) {
        for (int i=0; i<numVertices; i++) {
           v[i].subtractRotation(xform);
        }
        normal.subtractRotation(xform);
    }

    /**
        Wylicza jednostkowy wektor normalny dla tego wielokta.
        Metoda ta wykorzystuje pierwszy, drugi i trzeci wierzchoek, 
        by wyliczy wektor normalny, wic jeli wierzchoki te le w 
        jednej linii, to metoda ta nie bdzie dziaa. W tym przypadku,
        mona otrzyma wektor normalny z bokw wielokta.
        Metoda setNormal() pozwala rcznie ustawi wektor normalny.
        Wykorzystuje ona do oblicze statyczne obiekty w klasie Polygon3D, 
        wic nie gwarantuje bezpieczestwa oblicze w przypadku wielu
        rwnolegle wykorzystywanych instancji klasy Polygon3D.
    */
    public Vector3D calcNormal() {
        if (normal == null) {
            normal = new Vector3D();
        }
        temp1.setTo(v[2]);
        temp1.subtract(v[1]);
        temp2.setTo(v[0]);
        temp2.subtract(v[1]);
        normal.setToCrossProduct(temp1, temp2);
        normal.normalize();
        return normal;
    }


    /**
        Pobiera wektor normalny tego wielokta. Jeli wierzchoki si
        zmieni, naley uy calcNormal().
    */
    public Vector3D getNormal() {
        return normal;
    }


    /**
        Definiuje wektor normalny dla tego wielokta.
    */
    public void setNormal(Vector3D n) {
        if (normal == null) {
            normal = new Vector3D(n);
        }
        else {
            normal.setTo(n);
        }
    }

    /**
        Sprawdza, czy wielokt zwrcony jest przodem do okrelonego punktu.
        Metoda ta wykorzystuje do oblicze statyczne obiekty w klasie Polygon3D, 
        wic nie gwarantuje bezpieczestwa oblicze w przypadku wielu
        rwnolegle wykorzystywanych instancji klasy Polygon3D.
    */
    public boolean isFacing(Vector3D u) {
        temp1.setTo(u);
        temp1.subtract(v[0]);
        return (normal.getDotProduct(temp1) >= 0);
    }

    /**
        Przycina ten wielokt w taki sposb, aby wszystkie jego
        wierzchoki znalazy si przed paszczyzn przycinania, clipZ 
        (czyli, aby ich wsprzdne z speniay warunek z <= cilpZ).
        Warto clipZ powinna by rna od 0, aby unikn problemw
        zwizanych z dzieleniem przez 0.
        Zwraca true (prawda), jeli wielokt przynajmniej czciowo
        znajduje si przed powierzchni przycinania.
    */
    public boolean clip(float clipZ) {
        ensureCapacity(numVertices * 3);

        boolean isCompletelyHidden = true;

        // wstawia nowe wierzchoki, dziki ktrym kada z krawdzi bdzie,
        // albo cakowicie przed, albo cakowicie za paszczyzn przycinania
        for (int i=0; i<numVertices; i++) {
            int next = (i + 1) % numVertices;
            Vector3D v1 = v[i];
            Vector3D v2 = v[next];
            if (v1.z < clipZ) {
                isCompletelyHidden = false;
            }
            // sprawd czy v1.z < v2.z
            if (v1.z > v2.z) {
                Vector3D temp = v1;
                v1 = v2;
                v2 = temp;
            }
            if (v1.z < clipZ && v2.z > clipZ) {
                float scale = (clipZ-v1.z) / (v2.z - v1.z);
                insertVertex(next,
                    v1.x + scale * (v2.x - v1.x) ,
                    v1.y + scale * (v2.y - v1.y),
                    clipZ);
                // pomi wierzchoek, ktry wanie utworzylimy
                i++;
            }
        }

        if (isCompletelyHidden) {
            return false;
        }

        // usu wszystkie wierzchoki, dla ktrych z > clipZ
        for (int i=numVertices-1; i>=0; i--) {
            if (v[i].z > clipZ) {
                deleteVertex(i);
            }
        }

        return (numVertices >= 3);
    }

    /**
        Wstawia nowy wierzchoek pod podanym indeksem.
    */
    protected void insertVertex(int index, float x, float y,
        float z)
    {
        Vector3D newVertex = v[v.length-1];
        newVertex.x = x;
        newVertex.y = y;
        newVertex.z = z;
        for (int i=v.length-1; i>index; i--) {
            v[i] = v[i-1];
        }
        v[index] = newVertex;
        numVertices++;
    }


    /**
        Usuwa wierzchoek o podanym indeksie.
    */
    protected void deleteVertex(int index) {
        Vector3D deleted = v[index];
        for (int i=index; i<v.length-1; i++) {
            v[i] = v[i+1];
        }
        v[v.length-1] = deleted;
        numVertices--;
    }

    /**
        Wstawia wierzchoek do wielokta pod podanym indeksem.
        Wstawiany jest dokadnie ten wierzchoek (a nie jego kopia).
    */
    public void insertVertex(int index, Vector3D vertex) {
        Vector3D[] newV = new Vector3D[numVertices+1];
        System.arraycopy(v,0,newV,0,index);
        newV[index] = vertex;
        System.arraycopy(v,index,newV,index+1,numVertices-index);
        v = newV;
        numVertices++;
    }

    /**
        Calculates and returns the smallest bounding rectangle for
        this polygon.
    */
    public Rectangle3D calcBoundingRectangle() {

        // najmniejszy prostokt ograniczajcy wielokt ma z nim 
        // wsplny przynajmniej jeden bok. Dlatego ta metoda odnajduje 
        // ograniczajcy prostokt dla kadej z krawdzi 
        // wielokta i zwraca najmniejszy z nich.
        Rectangle3D boundingRect = new Rectangle3D();
        float minimumArea = Float.MAX_VALUE;
        Vector3D u = new Vector3D();
        Vector3D v = new Vector3D();
        Vector3D d = new Vector3D();
        for (int i=0; i<getNumVertices(); i++) {
            u.setTo(getVertex((i + 1) % getNumVertices()));
            u.subtract(getVertex(i));
            u.normalize();
            v.setToCrossProduct(getNormal(), u);
            v.normalize();

            float uMin = 0;
            float uMax = 0;
            float vMin = 0;
            float vMax = 0;
            for (int j=0; j<getNumVertices(); j++) {
                if (j != i) {
                    d.setTo(getVertex(j));
                    d.subtract(getVertex(i));
                    float uLength = d.getDotProduct(u);
                    float vLength = d.getDotProduct(v);
                    uMin = Math.min(uLength, uMin);
                    uMax = Math.max(uLength, uMax);
                    vMin = Math.min(vLength, vMin);
                    vMax = Math.max(vLength, vMax);
                }
            }
            // jeli ten wyliczony obszar jest najmniejszy, 
            // wybierz wanie ten ograniczajcy prostokt.
            float area = (uMax - uMin) * (vMax - vMin);
            if (area < minimumArea) {
                minimumArea = area;
                Vector3D origin = boundingRect.getOrigin();
                origin.setTo(getVertex(i));
                d.setTo(u);
                d.multiply(uMin);
                origin.add(d);
                d.setTo(v);
                d.multiply(vMin);
                origin.add(d);
                boundingRect.getDirectionU().setTo(u);
                boundingRect.getDirectionV().setTo(v);
                boundingRect.setWidth(uMax - uMin);
                boundingRect.setHeight(vMax - vMin);
            }
        }
        return boundingRect;
    }
}
