package com.brackeen.javagamebook.graphics3D;

import java.awt.*;
import java.awt.image.*;
import com.brackeen.javagamebook.math3D.*;
import com.brackeen.javagamebook.graphics3D.texture.*;

/**
    Klasa ShadedTexturedPolygonRenderer jest podklas PolygonRenderer
    renderujc dynamicznie tekstur ShadedTexture owietlon jednym 
    wiatem. Domylnie intensywno wiata otoczenia wynosi 0.5 i nie
    ma adnego wiata punktowego.
*/
public class ShadedTexturedPolygonRenderer
    extends FastTexturedPolygonRenderer
{

    private PointLight3D lightSource;
    private float ambientLightIntensity = 0.5f;
    private Vector3D directionToLight = new Vector3D();

    public ShadedTexturedPolygonRenderer(Transform3D camera,
        ViewWindow viewWindow)
    {
        this(camera, viewWindow, true);
    }

    public ShadedTexturedPolygonRenderer(Transform3D camera,
        ViewWindow viewWindow, boolean clearViewEveryFrame)
    {
        super(camera, viewWindow, clearViewEveryFrame);
    }


    /**
        Pobiera punktowe rdo wiata dla tego renderera.
    */
    public PointLight3D getLightSource() {
        return lightSource;
    }


    /**
        Ustawia rdo wiata dla tego renderera.
    */
    public void setLightSource(PointLight3D lightSource) {
        this.lightSource = lightSource;
    }


    /**
        Pobiera intensywno wiata otoczenia.
    */
    public float getAmbientLightIntensity() {
        return ambientLightIntensity;
    }


    /**
        Ustawia intensywno wiata otoczenia, zasadniczo midzy 0 a
        1.
    */
    public void setAmbientLightIntensity(float i) {
        ambientLightIntensity = i;
    }


    protected void drawCurrentPolygon(Graphics2D g) {
        // ustaw poziom cieniowania wielokta, zanim zaczniesz rysowa
        if (sourcePolygon instanceof TexturedPolygon3D) {
            TexturedPolygon3D poly =
                ((TexturedPolygon3D)sourcePolygon);
            Texture texture = poly.getTexture();
            if (texture instanceof ShadedTexture) {
                calcShadeLevel();
            }
        }
        super.drawCurrentPolygon(g);
    }


    /**
        Wylicza poziom cieniowania dla biecego wielokta
    */
    private void calcShadeLevel() {
        TexturedPolygon3D poly = (TexturedPolygon3D)sourcePolygon;
        float intensity = 0;
        if (lightSource != null) {


            // punkt bdcy redni z wierzchokw wielokta
            directionToLight.setTo(0,0,0);
            for (int i=0; i<poly.getNumVertices(); i++) {
                directionToLight.add(poly.getVertex(i));
            }
            directionToLight.divide(poly.getNumVertices());

            // utwrz wektor wychodzcy z tak urednionego wierzchoka
            // skierowany w kierunku wiata
            directionToLight.subtract(lightSource);
            directionToLight.multiply(-1);

            // pobierz odlego od rda wiata dla potrzeb wygaszania
            float distance = directionToLight.length();

            // wylicz odbicie dyfuzyjne 
            directionToLight.normalize();
            Vector3D normal = poly.getNormal();
            intensity = lightSource.getIntensity(distance)
                * directionToLight.getDotProduct(normal);
            intensity = Math.min(intensity, 1);
            intensity = Math.max(intensity, 0);
        }

        intensity+=ambientLightIntensity;
        intensity = Math.min(intensity, 1);
        intensity = Math.max(intensity, 0);
        int level =
            Math.round(intensity*ShadedTexture.MAX_LEVEL);
        ((ShadedTexture)poly.getTexture()).
            setDefaultShadeLevel(level);
    }

}