package com.brackeen.javagamebook.bsp2D;

import java.awt.geom.*;
import com.brackeen.javagamebook.math3D.*;

public class BSPLine extends Line2D.Float {

    public static final int BACK = -1;
    public static final int COLLINEAR = 0;
    public static final int FRONT = 1;
    public static final int SPANNING = 2;

    /**
        Wsprzdna X wektora prostopadego.
    */
    public float nx;

    /**
        Wsprzdna Y wektora prostopadego.
    */
    public float ny;

    /**
        Najwysze pooenie linii reprezentujcej cian.
    */
    public float top;

    /**
        Najnisze pooenie linii reprezentujcej cian.
    */
    public float bottom;

    /**
        Tworzy now lini z punktu (0,0) do punktu (0,0)
    */
    public BSPLine() {
        super();
    }


    /**
        Tworzy nowy obiekt BSPLine dla przekazanego obiektu BSPPolygon
        (tylko jeli obiekt BSPPolygon reprezentuje pionow cian).
    */
    public BSPLine(BSPPolygon poly) {
        setTo(poly);
    }


    /**
        Tworzy nowy obiekt BSPLine dla przekazanych wsprzdnych.
    */
    public BSPLine(float x1, float y1, float x2, float y2) {
        setLine(x1, y1, x2, y2);
    }


    /**
        czy ten obiekt BSPLine z danym obiektem BSPPolygon
        (tylko jeli obiekt BSPPolygon reprezentuje pionow cian).
    */
    public void setTo(BSPPolygon poly) {
        if (!poly.isWall()) {
            throw new IllegalArgumentException(
                "BSPPolygon not a wall");
        }
        top = java.lang.Float.MIN_VALUE;
        bottom = java.lang.Float.MAX_VALUE;
        // znajduje par najbardziej oddalonych punktw (ignorujc wsprzdn y)
        float distance = -1;
        for (int i=0; i<poly.getNumVertices(); i++) {
            Vector3D v1 = poly.getVertex(i);
            top = Math.max(top, v1.y);
            bottom = Math.min(bottom, v1.y);
            for (int j=0; j<poly.getNumVertices(); j++) {
                Vector3D v2 = poly.getVertex(j);
                float newDist = (float)Point2D.distanceSq(
                    v1.x, v1.z, v2.x, v2.z);
                if (newDist > distance) {
                    distance = newDist;
                    x1 = v1.x;
                    y1 = v1.z;
                    x2 = v2.x;
                    y2 = v2.z;
                }
            }
        }
        nx = poly.getNormal().x;
        ny = poly.getNormal().z;
    }

    /**
        Oblicza dane prostej prostopadej do tej linii.
    */
    public void calcNormal() {
        nx = y2 - y1;
        ny = x1 - x2;
    }

    /**
        Normalizuje dugo wektora prostopadego do linii podziau (zmienia
        jego dugo na 1 jednostk).
    */
    public void normalize() {
        float length = (float)Math.sqrt(nx * nx + ny * ny);
        nx/=length;
        ny/=length;
    }


    public void setLine(float x1, float y1, float x2, float y2) {
        super.setLine(x1, y1, x2, y2);
        calcNormal();
    }


    public void setLine(double x1, double y1, double x2,
        double y2)
    {
        super.setLine(x1, y1, x2, y2);
        calcNormal();
    }


    /**
        Zamienia punkty kocowe tej linii (innymi sowy, (x1,y1) staje si (x2,y2)
        i odwrotnie) oraz odwraca kierunek wskazywany przez prostpad.
    */
    public void flip() {
        float tx = x1;
        float ty = y1;
        x1 = x2;
        y1 = y2;
        x2 = tx;
        y2 = ty;
        nx = -nx;
        ny = -ny;
    }


    /**
        Ustawia wysoko na jakiej znajduje si dolna i grna krawd "ciany".
    */
    public void setHeight(float top, float bottom) {
        this.top = top;
        this.bottom = bottom;
    }


    /**
        Zwraca true, jeli punkty kocowe tej linii pokrywaj si z punktami
        kocowymi przekazanej linii. Ignoruje prostopad i wartoci dotyczce
        wysokoci.
    */
    public boolean equals(BSPLine line) {
        return (x1 == line.x1 && x2 == line.x2 &&
            y1 == line.y1 && y2 == line.y2);
    }


    /**
        Zwraca true, jeli punkty kocowe tej linii pokrywaj si z punktami
        kocowymi przekazanej linii. Ignoruje kolejno punktw kocowych
        (jeli pierwszy punkt kocowy tej linii pokrywa si z drugim punktem
        kocowym przekazanej linii, i odwrotnie, metoda zwraca true). Ignoruje
        take prostopad i wartoci dotyczce wysokoci.
    */
    public boolean equalsIgnoreOrder(BSPLine line) {
        return equals(line) || (
            (x1 == line.x2 && x2 == line.x1 &&
            y1 == line.y2 && y2 == line.y1));
    }


    public String toString() {
        return "(" + x1 + ", " + y1 + ")->" +
            "(" + x2 + "," + y2 + ")" +
            " bottom: " + bottom + " top: " + top;
    }


    /**
        Zwraca stron, po ktrej znajduje si okrelony punkt wzgldem linii
        podziau. Metoda zakada, e linia podziau ma grubo jednej jednostki,
        zatem punkty znajdujce si w granicach tej pogrubionej linii s traktowane
        jak punkty nalece do linii podziau. Aby ta metoda dziaaa prawidowo, 
        naley znormalizowa dugo wektora prostopadego do linii podziau (albo 
        wic t lini z wieloktem, albo wywoujc metod normalize()).
        W zalenoci od pooenia punktu, metoda zwraca warto FRONT, BACK lub 
        COLLINEAR.
    */
    public int getSideThick(float x, float y) {
        int frontSide = getSideThin(x-nx/2, y-ny/2);
        if (frontSide == FRONT) {
            return FRONT;
        }
        else if (frontSide == BACK) {
            int backSide = getSideThin(x+nx/2, y+ny/2);
            if (backSide == BACK) {
                return BACK;
            }
        }
        return COLLINEAR;
    }


    /**
        Zwraca stron, po ktrej znajduje si okrelony punkt wzgldem linii
        podziau. Ze wzgldu na niedokadno oblicze na liczbach zmiennoprze-
        cinkowych, przypadki wykrycia przynalenoci punktu do linii podziau
        s bardzo rzadkie. Aby ta metoda dziaaa prawidowo, naley znormalizowa
        dugo wektora prostopadego do linii podziau (albo wic t lini z
        wieloktem, albo wywoujc metod normalize()).
        W zalenoci od pooenia punktu, metoda zwraca warto FRONT, BACK lub 
        COLLINEAR.
    */
    public int getSideThin(float x, float y) {
        // il. skal. wektora prowadzcego do punktu i w. prostopadego do l. podziau
        float side = (x - x1)*nx + (y - y1)*ny;
        return (side < 0)?BACK:(side > 0)?FRONT:COLLINEAR;
    }


    /**
        Zwraca stron (wzgldem linii podziau), po ktrej znajduje si
        przekazany odcinek. Zwraca warto FRONT, BACK, COLINEAR lub SPANNING.
    */
    public int getSide(Line2D.Float segment) {
        if (this == segment) {
            return COLLINEAR;
        }
        int p1Side = getSideThick(segment.x1, segment.y1);
        int p2Side = getSideThick(segment.x2, segment.y2);
        if (p1Side == p2Side) {
            return p1Side;
        }
        else if (p1Side == COLLINEAR) {
            return p2Side;
        }
        else if (p2Side == COLLINEAR) {
            return p1Side;
        }
        else {
            return SPANNING;
        }
    }


    /**
        Zwraca stron (wzgldem linii podziau), po ktrej znajduje si
        przekazany wielokt. Zwraca warto FRONT, BACK, COLINEAR lub 
        SPANNING.
    */
    public int getSide(BSPPolygon poly) {
        boolean onFront = false;
        boolean onBack = false;

        // sprawdza kady punkt
        for (int i=0; i<poly.getNumVertices(); i++) {
            Vector3D v = poly.getVertex(i);
            int side = getSideThick(v.x, v.z);
            if (side == BSPLine.FRONT) {
                onFront = true;
            }
            else if (side == BSPLine.BACK) {
                onBack = true;
            }
        }

        // klasyfikuje wielomian
        if (onFront && onBack) {
            return BSPLine.SPANNING;
        }
        else if (onFront) {
            return BSPLine.FRONT;
        }
        else if (onBack) {
            return BSPLine.BACK;
        }
        else {
            return BSPLine.COLLINEAR;
        }
    }


    /**
        Zwraca stosunek dugoci jednej z czci przecitego odcinka
        do dugoci caego odcinka. Jeli odcinki si przecinaj,
        zwraca warto od 0 do 1. Przykadowo, zwrcona warto 0
        oznacza, e przecicie nastpio w punkcie (x1, y1); warto
        1 oznacza, e przecicie nastpio w punkcie (x2, y2);
        natomiast warto 0,5 oznacza, e przecicie nastpio w rodku
        odcinka. Zwraca warto -1, jeli odcinki s rwnolege.
    */
    public float getIntersection(Line2D.Float line) {
        // Punkt przecicia I dwch wektorw, A1->A2 i B1->B2:
        // I = A1 + u * (A2 - A1)
        // I = B1 + v * (B2 - B1)
        //
        // Rozwizanie dla u daje nam nastpujcy wzr.
        // Wynikiem rwnania jest warto u.
        float denominator = (line.y2 - line.y1) * (x2 - x1) - (line.x2 - line.x1) * (y2 - y1);

        // Sprawdza, czy dwie badane proste s rwnolege
        if (denominator == 0) {
            return -1;
        }

        float numerator = (line.x2 - line.x1) * (y1 - line.y1) - (line.y2 - line.y1) * (x1 - line.x1);

        return numerator / denominator;
    }


    /**
        Zwraca wsprzdne punktu przecicia danej prostej i prostej
        przekazanej w formie argumentu.
    */
    public Point2D.Float getIntersectionPoint(Line2D.Float line) {
        return getIntersectionPoint(line, null);
    }


    /**
        Zwraca wsprzdne punktu przecicia danej prostej i prostej
        przekazanej w formie argumentu. Jeli argument intersection
        ma warto null, tworzony jest nowy punkt.
    */
    public Point2D.Float getIntersectionPoint(Line2D.Float line,
        Point2D.Float intersection)
    {
        if (intersection == null) {
            intersection = new Point2D.Float();
        }
        float fraction = getIntersection(line);
        intersection.setLocation(
            x1 + fraction * (x2 - x1),
            y1 + fraction * (y2 - y1));
        return intersection;
    }

}
