package com.brackeen.javagamebook.util;
import java.util.LinkedList;

/**
    Pula wtkw stanowi grup z okrelon liczb wtkw,
    ktre mog by uywane do wykonywania zada.
*/
public class ThreadPool extends ThreadGroup {

    private boolean isAlive;
    private LinkedList taskQueue;
    private int threadID;
    private static int threadPoolID;

    /**
        Tworzy nowy obiekt ThreadPool.
        @param numThreads Liczba wtkw w puli.
    */
    public ThreadPool(int numThreads) {
        super("ThreadPool-" + (threadPoolID++));
        setDaemon(true);

        isAlive = true;

        taskQueue = new LinkedList();
        for (int i=0; i<numThreads; i++) {
            new PooledThread().start();
        }
    }


    /**
        Powoduje uruchomienie nowego zadania. Metoda ta natychmiast si koczy
        a zadanie jest wykonywane przez nastpny wolny wtek z ThreadPool
        <p>Zadania s wykonywane w kolejnoci ich uruchomienia.
        @param task Zadanie do wykonania. Jeeli ma warto null, nie jest
        podejmowana adna akcja.
        @throws IllegalStateException - w przypadku, gdy ten ThreadPool jest
        zamknity.
    */
    public synchronized void runTask(Runnable task) {
        if (!isAlive) {
            throw new IllegalStateException();
        }
        if (task != null) {
            taskQueue.add(task);
            notify();
        }

    }


    protected synchronized Runnable getTask()
        throws InterruptedException
    {
        while (taskQueue.size() == 0) {
            if (!isAlive) {
                return null;
            }
            wait();
        }
        return (Runnable)taskQueue.removeFirst();
    }


    /**
        Zamyka ten obiekt ThreadPool i koczy prac. Wszystkie zadania s 
        zatrzymywane i nie zostanie uruchomiony aden wtek oczekujcy.
        Po zamkniciu obiektu ThreadPool, nie moe on ju uruchamia
        adnych zada.
    */
    public synchronized void close() {
        if (isAlive) {
            isAlive = false;
            taskQueue.clear();
            interrupt();
        }
    }


    /**
        Zamyka tan obiekt ThreadPool i czeka na zakoczenie wszystkich 
        dziaajcych wtkw. Wykonywane s wszystkie oczekujce wtki.
    */
    public void join() {
        // powiadamia wszystkie oczekujce wtki, e biecy ThreadPool
        // zosta wyczony.
        synchronized (this) {
            isAlive = false;
            notifyAll();
        }

        // oczekiwanie na zakoczenie wszystkich wtkw
        Thread[] threads = new Thread[activeCount()];
        int count = enumerate(threads);
        for (int i=0; i<count; i++) {
            try {
                threads[i].join();
            }
            catch (InterruptedException ex) { }
        }
    }

    /**
        Sygnalizuje, e PooledThread zosta uruchomiony. 
        Metoda ta nic nie robi; specyficzne operacje powinny by zdefiniowane
        w klasach pochodnych.
    */
    protected void threadStarted() {
        // nic nie rb
    }


    /**
        Sygnalizuje, e PooledThread zosta zatrzymany.
        Metoda ta nic nie robi; specyficzne operacje powinny by zdefiniowane
        w klasach pochodnych.
    */
    protected void threadStopped() {
        // nic nie rb
    }


    /**
        PooledThread jest wtkiem z grupy ThreadPool, ktry moe
        wykonywa zadania (Runnable).
    */
    private class PooledThread extends Thread {


        public PooledThread() {
            super(ThreadPool.this,
                "PooledThread-" + (threadID++));
        }


        public void run() {
            // sygnalizacja, e wtek zosta uruchomiony
            threadStarted();

            while (!isInterrupted()) {

                // pobranie zadania do wykonania
                Runnable task = null;
                try {
                    task = getTask();
                }
                catch (InterruptedException ex) { }

                // jeeli getTask() zwrci null lub zosta przerwany,
                // wtek zostanie zakmnity.
                if (task == null) {
                    break;
                }

                // uruchamia wtek i zbiera wszystkie zgoszone 
                // przez niego wyjtki
                try {
                    task.run();
                }
                catch (Throwable t) {
                    uncaughtException(this, t);
                }
            }
            // sygnalizacja, e wtek zosta zatrzymany
            threadStopped();
        }
    }
}