package com.brackeen.javagamebook.sound;

/**
    Klasa EchoFilter dziedziczy po SoundFilter i emuluje efekt echa.
    @see FilteredSoundStream
*/
public class EchoFilter extends SoundFilter {

    private short[] delayBuffer;
    private int delayBufferPos;
    private float decay;

    /**
        Tworzenie EchoFilter z okrelon liczb opnionych prbek
        oraz wytumieniem.
        <p>Ilo opnionych prbek okrela czas po ktrym jest
        syszalne pierwsze echo. Dla echa o opnieniu 1 sekundy
        dla dwiku mono, 44100 Hz, naley uy wartoci 44100.
        <p>Warto wytumienia oznacza stopie wyciszenia echa.
        Warto 0,5 oznacza, e echo jest syszane z poow
        gonoci.
    */
    public EchoFilter(int numDelaySamples, float decay) {
        delayBuffer = new short[numDelaySamples];
        this.decay = decay;
    }


    /**
        Odczytanie iloci pozostaych danych generowanych przez filtr
        po zakoczeniu odtwarzania oryginalnego dwiku.
        Zapewnia podtrzymanie dwiku do 1%
        oryginalnej gonoci (amplitudy).
    */
    public int getRemainingSize() {
        float finalDecay = 0.01f;
        // pobrane z Math.pow(decay,x) <= finalDecay
        int numRemainingBuffers = (int)Math.ceil(
            Math.log(finalDecay) / Math.log(decay));
        int bufferSize = delayBuffer.length * 2;

        return bufferSize * numRemainingBuffers;
    }

    /**
        Kasuje wewntrzne bufory opniajce obiektu EchoFilter.
    */
    public void reset() {
        for (int i=0; i<delayBuffer.length; i++) {
            delayBuffer[i] = 0;
        }
        delayBufferPos = 0;
    }

    /**
        Filtruje prbki dwikowe w celu dodania echa. Odtwarzane prbki
        s dodawane do dwiku z bufora opnienia pomnoonego o
        warto wyguszania. Wynik jest rwnie zapisywany w
        buforze opnienia, wic mona usysze wielokrotne echo.
    */
    public void filter(byte[] samples, int offset, int length) {

        for (int i=offset; i<offset+length; i+=2) {
            // aktualizacja prbek
            short oldSample = getSample(samples, i);
            short newSample = (short)(oldSample + decay *
                delayBuffer[delayBufferPos]);
            setSample(samples, i, newSample);

            // aktualizacja bufora opnienia
            delayBuffer[delayBufferPos] = newSample;
            delayBufferPos++;
            if (delayBufferPos == delayBuffer.length) {
                delayBufferPos = 0;
            }
        }
    }
}
