package com.brackeen.javagamebook.game;

import java.awt.Rectangle;
import java.awt.Graphics2D;
import java.util.List;
import java.util.ArrayList;
import com.brackeen.javagamebook.math3D.*;

/**
    Klasa GridGameObjectManager jest implementacj interfejsu GameObjectManager,
    w ktrej obiekty GameObject s rozdzielan w dwuwymiarowej siatce, co pozwala
    okreli widoczno i ograniczy liczb testw majcych na celu wykrywanie
    kolizji.
*/
public class GridGameObjectManager implements GameObjectManager {

    /**
        Domylny rozmiar siatki to 512. Rozmiar powinien by wikszy dla wikszych
        obiektw.
    */
    private static final int GRID_SIZE_BITS = 9;
    private static final int GRID_SIZE = 1 << GRID_SIZE_BITS;

    /**
        Klasa Cell reprezentuje komrk w siatce. Zawiera list obiektw w grze
        i flag widocznoci.
    */
    private static class Cell {
        List objects;
        boolean visible;

        Cell() {
            objects = new ArrayList();
            visible = false;
        }
    }

    private Cell[] grid;
    private Rectangle mapBounds;
    private int gridWidth;
    private int gridHeight;
    private List allObjects;
    private List spawnedObjects;
    private GameObject player;
    private Vector3D oldLocation;
    private CollisionDetection collisionDetection;

    /**
        Tworzy nowy obiekt klasy GridGameObjectManager z podanymi
        ograniczeniami mapy i obiektem obsugujcym wykrywanie kolizji.
        Obiekty GameObjects spoza ogranicze mapy nie bd nigdy
        wywietlane.
    */
    public GridGameObjectManager(Rectangle mapBounds,
        CollisionDetection collisionDetection)
    {
        this.mapBounds = mapBounds;
        this.collisionDetection = collisionDetection;
        gridWidth = (mapBounds.width >> GRID_SIZE_BITS) + 1;
        gridHeight = (mapBounds.height >> GRID_SIZE_BITS) + 1;
        grid = new Cell[gridWidth*gridHeight];
        for (int i=0; i<grid.length; i++) {
            grid[i] = new Cell();
        }
        allObjects = new ArrayList();
        spawnedObjects = new ArrayList();
        oldLocation = new Vector3D();
    }


    /**
        Przeksztaca wsprzdne x mapy we wsprzdne x siatki.
    */
    private int convertMapXtoGridX(int x) {
        return (x - mapBounds.x) >> GRID_SIZE_BITS;
    }


    /**
        Przeksztaca wsprzdne y mapy we wsprzdne y siatki.
    */
    private int convertMapYtoGridY(int y) {
        return (y - mapBounds.y) >> GRID_SIZE_BITS;
    }


    /**
        Oznacza wszystkie obiekty jako potencjalnie widoczne (czyli takie,
        ktre naley narysowa).
    */
    public void markAllVisible() {
        for (int i=0; i<grid.length; i++) {
            grid[i].visible = true;
        }
    }


    /**
        Oznacza wszystkie obiekty wewntrz przekazanego obszaru 2D jako
        potencjalnie widoczne (czyli takie, ktre naley narysowa).
    */
    public void markVisible(Rectangle bounds) {
        int x1 = Math.max(0, convertMapXtoGridX(bounds.x));
        int y1 = Math.max(0, convertMapYtoGridY(bounds.y));
        int x2 = Math.min(gridWidth-1,
            convertMapXtoGridX(bounds.x + bounds.width));
        int y2 = Math.min(gridHeight-1,
            convertMapYtoGridY(bounds.y + bounds.height));

        for (int y=y1; y<=y2; y++) {
            int offset = y * gridWidth;
            for (int x=x1; x<=x2; x++) {
                grid[offset+x].visible = true;
            }
        }
    }


    /**
        Dodaje obiekt GameObject do tego menadera.
    */
    public void add(GameObject object) {
        if (object != null) {
            if (object == player) {
                // zapewnia, e ruch gracza nastpuje jako pierwszy
                allObjects.add(0, object);
            }
            else {
                allObjects.add(object);
            }
            Cell cell = getCell(object);
            if (cell != null) {
                cell.objects.add(object);
            }

        }
    }


    /**
        Usuwa obiekt GameObject z tego menadera.
    */
    public void remove(GameObject object) {
        if (object != null) {
            allObjects.remove(object);
            Cell cell = getCell(object);
            if (cell != null) {
                cell.objects.remove(object);
            }
        }
    }


    /**
        Dodaje obiekt GameObject do tego menadera, zaznaczajc, e
        jest to obiekt gracza. Istniejcy obiekt gracza, jeli wystpuje,
        nie jest usuwany.
    */
    public void addPlayer(GameObject player) {
        this.player = player;
        if (player != null) {
            player.notifyVisible(true);
            add(player);
        }
    }


    /**
        Zwraca obiekt oznaczony jako obiekt gracza lub null,
        jeli nie wskazano obiektu gracza.
    */
    public GameObject getPlayer() {
        return player;
    }


    /**
        Zwraca komrk, w ktrej znajduje si podany obiekt GameObject, lub
        null, jeli ten obiekt GameObject nie mieci si w granicach mapy.
    */
    private Cell getCell(GameObject object) {
        int x = convertMapXtoGridX((int)object.getX());
        int y = convertMapYtoGridY((int)object.getZ());
        return getCell(x, y);
    }


    /**
        Zwraca komrk, w ktrej znajduje si podany punkt, lub
        null, jeli wsprzdne punktu nie s prawidowe.
    */
    private Cell getCell(int x, int y) {

        // sprawdza ograniczenia
        if (x < 0 || y < 0 || x >= gridWidth || y >= gridHeight) {
            return null;
        }

        // zwraca komrk, do ktrej naley punkt o wsprzdnych x, y
        return grid[x + y * gridWidth];
    }


    /**
        Aktualizuje wszystkie obiektu na podstawie czasu, ktry upyn
        od ostatniej aktualizacji i zastosowania mechanizmu wykrywania
        kolizji.
    */
    public void update(long elapsedTime) {
        for (int i=0; i<allObjects.size(); i++) {
            GameObject object = (GameObject)allObjects.get(i);

            // zapisz star pozycj obiektu
            Cell oldCell = getCell(object);
            oldLocation.setTo(object.getLocation());
            boolean isRegenerating = false;

            // przesu obiekt
            object.update(player, elapsedTime);

            // led wszystkie nowe obiekty (dodawane za chwil)
/*            List spawns = object.getSpawns();
            if (spawns != null) {
                if (spawns.contains(object)) {
                    isRegenerating = true;
                }
                spawnedObjects.addAll(spawns);
            }
*/
            // jeli obiekt zosta zniszczony, usu go
            if (object.isDestroyed() || isRegenerating) {
                allObjects.remove(i);
                i--;
                if (oldCell != null) {
                    oldCell.objects.remove(object);
                }
                continue;
            }

            // jeli obiekt wykona ruch, wykonaj procedur wykrywania kolizji
            if (!object.getLocation().equals(oldLocation) ||
                object.isJumping())
            {

                // sprawd ciany, podogi i sufity
                collisionDetection.checkBSP(object,
                    oldLocation, elapsedTime);

                // sprawd pozostae obiekty
                if (checkObjectCollision(object, oldLocation)) {
                    // revert to old position
                    object.getLocation().setTo(oldLocation);
                }

                // zaktualizuj pooenie w siatce
                Cell cell = getCell(object);
                if (cell != oldCell) {
                    if (oldCell != null) {
                        oldCell.objects.remove(object);
                    }
                    if (cell != null) {
                        cell.objects.add(object);
                    }
                }
            }

        }

        // dodaj wszystkie nowe obiekty
        if (spawnedObjects.size() > 0) {
            for (int i=0; i<spawnedObjects.size(); i++) {
                add((GameObject)spawnedObjects.get(i));
            }
            spawnedObjects.clear();
        }
    }


    /**
        Sprawdza, czy dany obiekt koliduje z ktrymkolwiek z pozostaych
        obiektw.
    */
    public boolean checkObjectCollision(GameObject object,
        Vector3D oldLocation)
    {

        boolean collision = false;

        // uyj wsprzdnych (x,z) dla obiektu (pozycja w poziomie)
        int x = convertMapXtoGridX((int)object.getX());
        int y = convertMapYtoGridY((int)object.getZ());

        // sprawd 9 komrek otaczajcych obiekt
        for (int i=x-1; i<=x+1; i++) {
            for (int j=y-1; j<=y+1; j++) {
                Cell cell = getCell(i, j);
                if (cell != null) {
                    collision |= collisionDetection.checkObject(
                        object, cell.objects, oldLocation);
                }
            }
        }

        return collision;
    }


    /**
        DNarysuj wszystkie widoczne obiekty i oznacz wszystkei obiekt,
        ktre nie s widoczne.
    */
    public void draw(Graphics2D g, GameObjectRenderer r) {
        for (int i=0; i<grid.length; i++) {
            List objects = grid[i].objects;
            for (int j=0; j<objects.size(); j++) {
                GameObject object = (GameObject)objects.get(j);
                boolean visible = false;
                if (grid[i].visible) {
                    visible = r.draw(g, object);
                }
                if (object != player) {
                    // poinformuj obiekty, ktre s widoczne
                    object.notifyVisible(visible);
                }
            }
            grid[i].visible = false;
        }
    }
}
