package com.brackeen.javagamebook.game;

import com.brackeen.javagamebook.math3D.Vector3D;

/**
    Klasa Physics reprezentuje rne atrybuty fizyczne (np. wpyw grawitacji)
    i udostpnia funkcje umoliwiajce modyfikowanie obiektw w oparciu o te
    atrybuty. Klasa obsuguje obecnie jedynie wpyw grawitacji i przyspieszenie
    podczas wchodzenia po schodach.
*/
public class Physics {

    /**
        Domylne przyspieszenie ziemskie w jednostkach na milisekund kwadratow
    */
    public static final float DEFAULT_GRAVITY_ACCEL = -.002f;

    /**
        Domylne przyspieszenie dla ruchu w gr schodw w jednostkach na
        milisekund kwadratow.
    */
    public static final float DEFAULT_SCOOT_ACCEL = .006f;

    private static Physics instance;

    private float gravityAccel;
    private float scootAccel;
    private Vector3D velocity = new Vector3D();

    /**
        Zwraca instancj klasy Phisics. Jeli taka instancja jeszcze
        nie istnieje, jest tworzona z domylnymi wartociami atrybutw.
    */
    public static synchronized Physics getInstance() {
        if (instance == null) {
            instance = new Physics();
        }
        return instance;
    }


    protected Physics() {
        gravityAccel = DEFAULT_GRAVITY_ACCEL;
        scootAccel = DEFAULT_SCOOT_ACCEL;
    }


    /**
        Zwraca przyspieszenie ziemskie w jednostkach na milisekund kwadratow.
    */
    public float getGravityAccel() {
        return gravityAccel;
    }


    /**
        Ustawia przyspieszenie ziemskie w jednostkach na milisekund kwadratow.
    */
    public void setGravityAccel(float gravityAccel) {
        this.gravityAccel = gravityAccel;
    }


    /**
        Zwraca przyspieszenie dla ruchu w gr schodw w jednostkach na
        milisekund kwadratow. Przyspieszenie to jest wykorzystywane do
        pynnego pokonywania schodw.
    */
    public float getScootAccel() {
        return scootAccel;
    }


    /**
        Ustawia przyspieszenie dla ruchu w gr schodw w jednostkach na
        milisekund kwadratow. Przyspieszenie to jest wykorzystywane do
        pynnego pokonywania schodw.
    */
    public void setScootAccel(float scootAccel) {
        this.scootAccel = scootAccel;
    }


    /**
        Stosuje przyspieszenie grawitacyjne dla okrelonego obiektu klasy
        GameObject wedug czasu, ktry min od ostatniej aktualizacji.
    */
    public void applyGravity(GameObject object, long elapsedTime) {
        velocity.setTo(0, gravityAccel * elapsedTime, 0);
        object.getTransform().addVelocity(velocity);
    }


    /**
        Stosuje przyspieszenie dla ruchu po schodach w gr dla okrelonego
        obiektu klasy GameObject wedug czasu, ktry min od ostatniej
        aktualizacji.
    */
    public void scootUp(GameObject object, long elapsedTime) {
        velocity.setTo(0, scootAccel * elapsedTime, 0);
        object.getTransform().addVelocity(velocity);
    }


    /**
        Stosuje odwrotne przyspieszenie dla ruchu po schodach w gr dla
        okrelonego obiektu klasy GameObject wedug czasu, ktry min od
        ostatniej aktualizacji.
    */
    public void scootDown(GameObject object, long elapsedTime) {
        velocity.setTo(0, -scootAccel * elapsedTime, 0);
        object.getTransform().addVelocity(velocity);
    }


    /**
        Ustawia dla danego obiektu GameObject wektor ruchu w gr, ktry
        umoliwi jego skok na okrelon wysoko. Wywouje metod
        getJumpVelocity(), ktra oblicza szybko ruchu za pomoc funkcji
        Math.sqrt().
    */
    public void jumpToHeight(GameObject object, float jumpHeight) {
        jump(object, getJumpVelocity(jumpHeight));
    }


    /**
        Ustawia okrelon szybko ruchu pionowego obiektu GameObject dla
        okrelonej szybkoci skoku.
    */
    public void jump(GameObject object, float jumpVelocity) {
        velocity.setTo(0, jumpVelocity, 0);
        object.getTransform().getVelocity().y = 0;
        object.getTransform().addVelocity(velocity);
    }


    /**
        Zwraca pocztkow szybko skoku potrzebn do osignicia
        okrelonej wysokoci (po uwzgldnieniu biecej grawitacji).
        Wykorzystuje w obliczeniach funkcj Math.sqrt().
    */
    public float getJumpVelocity(float jumpHeight) {
        // uyj wzoru dla szybkoci i przyspieszenia: v*v = -2 * a(y-y0)
        // (v jest szybkoci skoku, a to przysp., y-y0 to maksymalna wys.)
        return (float)Math.sqrt(-2*gravityAccel*jumpHeight);
    }
}
