package com.brackeen.javagamebook.bsp2D;


import java.util.*;
import com.brackeen.javagamebook.math3D.Vector3D;
import com.brackeen.javagamebook.path.AStarNode;

/**
    Klasa Portal reprezentuje rozdzielnik pomidzy dwoma
    limi drzewa BSP (np. przejcia pomidzy pomieszczeniami).
    Klasa Portal jest rozszerzeniem klasy AStarNode, zatem
    boty wyposaone w system sztucznej inteligencji bd mogy
    wykorzystywa algorytm A* do znajdowania drogi w drzewie
    BSP.
*/
public class Portal extends AStarNode {

    private BSPLine divider;
    private BSPTree.Leaf front;
    private BSPTree.Leaf back;
    private ArrayList neighbors;
    private Vector3D midPoint;

    /**
        Tworzy nowy obiekt klasy Portal z danym rozdzielnikiem i
        limi dla przodu i tyu.
    */
    public Portal(BSPLine divider, BSPTree.Leaf front,
        BSPTree.Leaf back)
    {
        this.divider = divider;
        this.front = front;
        this.back = back;
        midPoint = new Vector3D(
            (divider.x1 + divider.x2) / 2,
            Math.max(front.floorHeight, back.floorHeight),
            (divider.y1 + divider.y2) / 2);
    }


    /**
        Zwraca punkt rodkowy z rozdzielnika przejcia.
    */
    public Vector3D getMidPoint() {
        return midPoint;
    }


    /**
        Buduje list ssiadw dla reprezentacji AStarNode.
        Ssiadami s przejcia znajdujcie si w przednich
        i tylnych liciach, wyczajc to przejcie.
    */
    public void buildNeighborList() {
        neighbors = new ArrayList();
        if (front != null) {
            neighbors.addAll(front.portals);
        }
        if (back != null) {
            neighbors.addAll(back.portals);
        }

        // dostosuj rozmiar, po czym usu odwoania do tego wza.
        // (zapewnia dodatkow przestrze na wywoania metogy addNeighbor()
        // bez zwikszania przestrzeni w tablicy)
        neighbors.trimToSize();
        while (neighbors.remove(this));
    }


    /**
        Dodaje wze ssiadujcy do listy ssiadw.
    */
    public void addNeighbor(AStarNode node) {
        if (neighbors == null) {
            buildNeighborList();
        }
        neighbors.add(node);
    }


    /**
        Usuwa wze ssiadujcy z listy ssiadw.
    */
    public void removeNeighbor(AStarNode node) {
        if (neighbors == null) {
            buildNeighborList();
        }
        neighbors.remove(node);
    }

    // metody z klasy AStarNode

    public float getCost(AStarNode node) {
        return getEstimatedCost(node);
    }


    public float getEstimatedCost(AStarNode node) {
        if (node instanceof Portal) {
            Portal other = (Portal)node;
            float dx = midPoint.x - other.midPoint.x;
            float dz = midPoint.z - other.midPoint.z;
            return (float)Math.sqrt(dx * dx + dz * dz);
        }
        else {
            return node.getEstimatedCost(this);
        }
    }


    public List getNeighbors() {
        if (neighbors == null) {
            buildNeighborList();
        }
        return neighbors;
    }

}