package com.brackeen.javagamebook.graphics;

import java.awt.*;
import java.awt.image.BufferStrategy;
import java.awt.image.BufferedImage;
import java.lang.reflect.InvocationTargetException;
import javax.swing.JFrame;

/**
    Klasa ScreenManager zarzdza inicjalizacj i
    wywietlaniem penoekranowych trybw graficznych.
*/
public class ScreenManager {

    private GraphicsDevice device;

    /**
        Tworzy nowy obiekt ScreenManager.
    */
    public ScreenManager() {
        GraphicsEnvironment environment =
            GraphicsEnvironment.getLocalGraphicsEnvironment();
        device = environment.getDefaultScreenDevice();
    }


    /**
        Zwraca list trybw graficznych dostpnych dla 
        domylnego urzdzenia w systemie.
    */
    public DisplayMode[] getCompatibleDisplayModes() {
        return device.getDisplayModes();
    }


    /**
        Zwraca pierwszy dostpny tryb graficzny z listy trybw.
        Zwraca null, jeeli aden tryb nie jest zgodny.
    */
    public DisplayMode findFirstCompatibleMode(
        DisplayMode modes[])
    {
        DisplayMode goodModes[] = device.getDisplayModes();
        for (int i = 0; i < modes.length; i++) {
            for (int j = 0; j < goodModes.length; j++) {
                if (displayModesMatch(modes[i], goodModes[j])) {
                    return modes[i];
                }
            }

        }

        return null;
    }


    /**
        Zwraca biecy tryb graficzny.
    */
    public DisplayMode getCurrentDisplayMode() {
        return device.getDisplayMode();
    }


    /**
        Sprawdza, czy dwa tryby wywietlania "pasuj do siebie". Dwa tryby
        pasuj jeeli maj t sam rozdzielczo, gbi kolorw i
        czstotliwo oraz czstotliwo odwieania. Gbia kolorw jest
        ignorowana, jeeli jeden z trybw na gbi kolorw ustawion jako 
        DisplayMode.BIT_DEPTH_MULTI. Podobnie, czstotliwo odwieania jest
        ignorowana, gdy jeden z trybw ma j ustawion jako
        DisplayMode.REFRESH_RATE_UNKNOWN.
    */
    public boolean displayModesMatch(DisplayMode mode1,
        DisplayMode mode2)

    {
        if (mode1.getWidth() != mode2.getWidth() ||
            mode1.getHeight() != mode2.getHeight())
        {
            return false;
        }

        if (mode1.getBitDepth() != DisplayMode.BIT_DEPTH_MULTI &&
            mode2.getBitDepth() != DisplayMode.BIT_DEPTH_MULTI &&
            mode1.getBitDepth() != mode2.getBitDepth())
        {
            return false;
        }

        if (mode1.getRefreshRate() !=
            DisplayMode.REFRESH_RATE_UNKNOWN &&
            mode2.getRefreshRate() !=
            DisplayMode.REFRESH_RATE_UNKNOWN &&
            mode1.getRefreshRate() != mode2.getRefreshRate())
         {
             return false;
         }

         return true;
    }


    /**
        Przechodzi w tryb penoekranowy i zmienia tryb graficzny.
        Jeeli podany tryb wywietlania jest null lub nie jest zgodny
        z tym urzdzeniem, lub system nie pozwala zmienia trybw,
        Wykorzystany jest biecy tryb graficzny.
        <p>
        Wykorzystany jest obiekt BufferStrategy z dwoma buforami.
    */
    public void setFullScreen(DisplayMode displayMode) {
        final JFrame frame = new JFrame();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setUndecorated(true);
        frame.setIgnoreRepaint(true);
        frame.setResizable(false);

        device.setFullScreenWindow(frame);

        if (displayMode != null &&
            device.isDisplayChangeSupported())
        {
            try {
                device.setDisplayMode(displayMode);
            }
            catch (IllegalArgumentException ex) { }
            // poprawka dla mac os x
            frame.setSize(displayMode.getWidth(),
                displayMode.getHeight());
        }
        // unikanie potencjalnego zakleszczenia w wersji 1.4.1_02
        try {
            EventQueue.invokeAndWait(new Runnable() {
                public void run() {
                    frame.createBufferStrategy(2);
                }
            });
        }
        catch (InterruptedException ex) {
            // ignorowane
        }
        catch (InvocationTargetException  ex) {
            // ignorowane
        }
    }


    /**
        Pobiera kontekst graficzny dla ekranu. 
        ScreenManager korzysta z podwjnego buforowania, wic aplikacje 
        musz wywoywa update() w celu pokazania narysowanej grafiki.
        <p>
        Aplikacja musi usuwa obiekt graficzny.
    */
    public Graphics2D getGraphics() {
        Window window = device.getFullScreenWindow();
        if (window != null) {
            BufferStrategy strategy = window.getBufferStrategy();
            return (Graphics2D)strategy.getDrawGraphics();
        }
        else {
            return null;
        }
    }


    /**
        Odrysowuje ekran.
    */
    public void update() {
        Window window = device.getFullScreenWindow();
        if (window != null) {
            BufferStrategy strategy = window.getBufferStrategy();
            if (!strategy.contentsLost()) {
                strategy.show();
            }
        }
        // Operacja Sync dziaajca na niektrych systemach
        // (W Linux, naprawia problem z kolejk zdarze)
        // Toolkit.getDefaultToolkit().sync();
    }


    /**
        Zwraca okno uywane obecnie w trybie penoekranowym.
        Zwraca null, gdy urzdzenie nie jest w trybie penoekranowym.
    */
    public JFrame getFullScreenWindow() {
        return (JFrame)device.getFullScreenWindow();
    }


    /**
        Zwraca szeroko okna uytego obecnie w trybie penoekranowym.
        Zwraca 0 jeeli urzdzenie nie jest w trybie penoekranowym.
    */
    public int getWidth() {
        Window window = device.getFullScreenWindow();
        if (window != null) {
            return window.getWidth();
        }
        else {
            return 0;
        }
    }


    /**
        Zwraca wysoko okna uytego obecnie w trybie penoekranowym.
        Zwraca 0 jeeli urzdzenie nie jest w trybie penoekranowym.
    */
    public int getHeight() {
        Window window = device.getFullScreenWindow();
        if (window != null) {
            return window.getHeight();
        }
        else {
            return 0;
        }
    }


    /**
        Przywraca poprzedni tryb wywietlania.
    */
    public void restoreScreen() {
        Window window = device.getFullScreenWindow();
        if (window != null) {
            window.dispose();
        }
        device.setFullScreenWindow(null);
    }


    /**
        Tworzy obiekt Image zgodny z biecym ekranem.
    */
    public BufferedImage createCompatibleImage(int w, int h,
        int transparancy)
    {
        Window window = device.getFullScreenWindow();
        if (window != null) {
            GraphicsConfiguration gc =
                window.getGraphicsConfiguration();
            return gc.createCompatibleImage(w, h, transparancy);
        }
        return null;
    }
}
