package com.brackeen.javagamebook.graphics3D;

import java.awt.Graphics2D;
import java.awt.Color;
import com.brackeen.javagamebook.math3D.*;

/**
    Klasa PolygonRenderer jest abstrakcyjn klas odpowiedzialn
    za transformowanie wieloktw i rysowanie ich na ekranie.
*/
public abstract class PolygonRenderer {

    protected ScanConverter scanConverter;
    protected Transform3D camera;
    protected ViewWindow viewWindow;
    protected boolean clearViewEveryFrame;
    protected Polygon3D sourcePolygon;
    protected Polygon3D destPolygon;

    /**
        Tworzy nowy obiekt PolygonRenderer dla podanych obiektw
        Transform3D (kamery) i ViewWindow (okna obrazu). Obraz jest
        oczyszczany w momencie przywoania funkcji startFrame().
    */
    public PolygonRenderer(Transform3D camera,
        ViewWindow viewWindow)
    {
        this(camera, viewWindow, true);
    }


    /**
        Tworzy nowy obiekt PolygonRenderer dla podanych obiektw
        Transform3D (kamery) i ViewWindow (okna obrazu). Jeli
        clearViewEveryFrame ma warto true, obraz bdzie oczyszczany
        w momencie przywoania funkcji startFrame().
    */
    public PolygonRenderer(Transform3D camera,
        ViewWindow viewWindow, boolean clearViewEveryFrame)
    {
        this.camera = camera;
        this.viewWindow = viewWindow;
        this.clearViewEveryFrame = clearViewEveryFrame;
        init();
    }


    /**
        Tworzy konwerter skanujcy i docelowy wielokt.
    */
    protected void init() {
        destPolygon = new Polygon3D();
        scanConverter = new ScanConverter(viewWindow);
    }


    /**
        Pobiera kamer wykorzystywan przez ten obiekt PolygonRenderer.
    */
    public Transform3D getCamera() {
        return camera;
    }


    /**
        Wskazuje pocztek renderowania klatki obrazu. Metoda ta
        powinna by przywoywana dla kadej klatki, zanim jeszcze 
        zostanie narysowany jakikolwiek wielokt.
    */
    public void startFrame(Graphics2D g) {
        if (clearViewEveryFrame) {
            g.setColor(Color.black);
            g.fillRect(viewWindow.getLeftOffset(),
                viewWindow.getTopOffset(),
                viewWindow.getWidth(), viewWindow.getHeight());
        }
    }


    /**
        Informuje o kocu renderowania klatki obrazu. Metoda ta
        powinna by przywoywana dla kadej klatki ju po narysowaniu
        wszystkich wieloktw.
    */
    public void endFrame(Graphics2D g) {
        // tymczasem nie rb nic.
    }


    /**
        Transformuje i rysuje wielokt.
    */
    public boolean draw(Graphics2D g, Polygon3D poly) {
        if (poly.isFacing(camera.getLocation())) {
            sourcePolygon = poly;
            destPolygon.setTo(poly);
            destPolygon.subtract(camera);
            boolean visible = destPolygon.clip(-1);
            if (visible) {
                destPolygon.project(viewWindow);
                visible = scanConverter.convert(destPolygon);
                if (visible) {
                    drawCurrentPolygon(g);
                    return true;
                }
            }
        }
        return false;
    }


    /**
        Rysuje biecy wielokt. W tym momencie biecy
        wielokt jest transformowany, przycinany, rzutowany,
        konwertowany na linie i rysowany.
    */
    protected abstract void drawCurrentPolygon(Graphics2D g);
}