package com.brackeen.javagamebook.tilegame;

import java.awt.*;
import java.util.Iterator;

import com.brackeen.javagamebook.graphics.Sprite;
import com.brackeen.javagamebook.tilegame.sprites.Creature;

/**
    Klasa TileMapRenderer realizuje rysowanie TileMap na ekranie.
    Rysuje wszystkie kafelki oraz opcjonalny rysunek ta
    wyrodkowany wzgldem bicego pooenia bohatera.

    <p>Jeeli szeroko ta jest mniejsza ni szeroko mapy kafelek,
    bdzie si wydawo, e rysunek ta porusza si wolniej,
    tworzc efekt paralaksy.

    <p>Dodatkowo, udostpnione s rwnie trzy metody statyczne do konwersji
    pikseli na wsprzdne kafelkw i odwrotnie.

    <p>Ten obiekt TileMapRender korzysta z kafelkw o szerokoci 64.
*/
public class TileMapRenderer {

    private static final int TILE_SIZE = 64;
    // rozmiar kafelka w bitach
    // Math.pow(2, TILE_SIZE_BITS) == TILE_SIZE
    private static final int TILE_SIZE_BITS = 6;

    private Image background;

    /**
        Konwersja wsprzdnych pikselowych na wsprzdne kafelek.
    */
    public static int pixelsToTiles(float pixels) {
        return pixelsToTiles(Math.round(pixels));
    }

    /**
        Konwersja wsprzdnych pikselowych na wsprzdne kafelek.
    */
    public static int pixelsToTiles(int pixels) {
        // wykorzystanie przesuwania do korygowania ujemnych wsprzdnych
        return pixels >> TILE_SIZE_BITS;

        // w przypadku kafelkw o rozmiarach nie bdcych potg dwjki
        // uycie funkcji floor:
        //return (int)Math.floor((float)pixels / TILE_SIZE);
    }

    /**
        Konwersja wsprzdnych kafelek na wspprzedne pikselowe.
    */
    public static int tilesToPixels(int numTiles) {
        // Uycie przesuwania nie jest konieczne.
        // Jest nieco szybsze, ale nie daje zbyt duo
        // przyspieszenia na nowoczesnych procesorach.
        return numTiles << TILE_SIZE_BITS;

        // jeeli rozmiar kafelkw nie jest potg dwch:
        //return numTiles * TILE_SIZE;
    }

    /**
        Ustawienie ta do narysowania.
    */
    public void setBackground(Image background) {
        this.background = background;
    }

    /**
        Rysowanie obiektu TileMap.
    */
    public void draw(Graphics2D g, TileMap map,
        int screenWidth, int screenHeight)
    {
        Sprite player = map.getPlayer();
        int mapWidth = tilesToPixels(map.getWidth());

        // ustawienie pozycji przewijania mapy
        // korzystajc z pooenia bohatera
        int offsetX = screenWidth / 2 -
            Math.round(player.getX()) - TILE_SIZE;
        offsetX = Math.min(offsetX, 0);
        offsetX = Math.max(offsetX, screenWidth - mapWidth);

        // pobranie przesunicia wsprzdnej y do narysowania wszystkich 
        // duszkw i kafelek
        int offsetY = screenHeight -
            tilesToPixels(map.getHeight());

        // w razie potrzeby rysowanie czarnego ta
        if (background == null ||
            screenHeight > background.getHeight(null))
        {
            g.setColor(Color.black);
            g.fillRect(0, 0, screenWidth, screenHeight);
        }

        // rysowanie przesunitego rysunku ta
        if (background != null) {
            int x = offsetX *
                (screenWidth - background.getWidth(null)) /
                (screenWidth - mapWidth);
            int y = screenHeight - background.getHeight(null);

            g.drawImage(background, x, y, null);
        }

        // rysowanie widocznych kafelkw
        int firstTileX = pixelsToTiles(-offsetX);
        int lastTileX = firstTileX +
            pixelsToTiles(screenWidth) + 1;
        for (int y=0; y<map.getHeight(); y++) {
            for (int x=firstTileX; x <= lastTileX; x++) {
                Image image = map.getTile(x, y);
                if (image != null) {
                    g.drawImage(image,
                        tilesToPixels(x) + offsetX,
                        tilesToPixels(y) + offsetY,
                        null);
                }
            }
        }
        // rysowanie bohatera
        g.drawImage(player.getImage(),
            Math.round(player.getX()) + offsetX,
            Math.round(player.getY()) + offsetY,
            null);

        // rysowanie duszkw
        Iterator i = map.getSprites();
        while (i.hasNext()) {
            Sprite sprite = (Sprite)i.next();
            int x = Math.round(sprite.getX()) + offsetX;
            int y = Math.round(sprite.getY()) + offsetY;
            g.drawImage(sprite.getImage(), x, y, null);

            // obudzenie przeciwnika gdy znajdzie si na ekranie
            if (sprite instanceof Creature &&
                x >= 0 && x < screenWidth)
            {
                ((Creature)sprite).wakeUp();
            }
        }
    }
}
