package com.brackeen.javagamebook.sound;

import com.brackeen.javagamebook.graphics.Sprite;

/**
    Klasa Filter3d dziedziczy po SoundFilter i realizuje efekt
    dwiku 3D. Dwik jest filtrowany w taki sposb, e
    cichnie wraz ze wzrostem odlegoci od suchacza.
    <p>Moliwe rozszerzenia tej klasy:
    <ul><li>przesuwanie dwiku midzy lewym i prawym gonikiem.
    </ul>
    @see FilteredSoundStream
*/
public class Filter3d extends SoundFilter {

    // Liczba prbek do przesunicia podczas zmiany gonoci.
    private static final int NUM_SHIFTING_SAMPLES = 500;

    private Sprite source;
    private Sprite listener;
    private int maxDistance;
    private float lastVolume;

    /**
        Tworzy nowy obiekt Filter3d z podanym duszkiem rda i suchacza.
        Zmiana pooenia duszka moe by dokonywana w czasie dziaania
        filtra.
        <p> Parametr maxDistance jest maksymaln odlegoci, z ktrej
        jest syszalny dwik.
    */
    public Filter3d(Sprite source, Sprite listener,
        int maxDistance)
    {
        this.source = source;
        this.listener = listener;
        this.maxDistance = maxDistance;
        this.lastVolume = 0.0f;
    }

    /**
        Filtruje dwik w taki sposb, e przy wikszej odlegoci jest cichszy.
    */
    public void filter(byte[] samples, int offset, int length) {

        if (source == null || listener == null) {
            // Brak danych do filtrowania  koniec.
            return;
        }

        // Obliczenie odlegoci suchacza od rda dwiku.
        float dx = (source.getX() - listener.getX());
        float dy = (source.getY() - listener.getY());
        float distance = (float)Math.sqrt(dx * dx + dy * dy);

        // Ustawienie gonoci od 0 (cisza) do 1.
        float newVolume = (maxDistance - distance) / maxDistance;
        if (newVolume <= 0) {
            newVolume = 0;
        }

        // Ustawienie gonoci prbek.
        int shift = 0;
        for (int i=offset; i<offset+length; i+=2) {

            float volume = newVolume;

            // Przesunicie od ostatniej gonoci do nowej.
            if (shift < NUM_SHIFTING_SAMPLES) {
                volume = lastVolume + (newVolume - lastVolume) *
                    shift / NUM_SHIFTING_SAMPLES;
                shift++;
            }

            // Zmiana gonoci prbki.
            short oldSample = getSample(samples, i);
            short newSample = (short)(oldSample * volume);
            setSample(samples, i, newSample);
        }

        lastVolume = newVolume;
    }
}
