import java.awt.*;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.io.*;
import javax.swing.*;
import javax.sound.sampled.AudioFormat;
import javax.sound.midi.Sequence;
import javax.sound.midi.Sequencer;

import com.brackeen.javagamebook.graphics.NullRepaintManager;
import com.brackeen.javagamebook.sound.*;
import com.brackeen.javagamebook.test.GameCore;

/**
    Klasa SoundManagerTest demonstruje dziaanie klasy
    SoundManager. Zrealizowane zosty nastpujce demonstracje:
    <ul>
    <li>Odtwarzanie sekwencji Midi.
    <li>Przeczanie cieki sekwencji Midi.
    <li>Odtwarzanie dwiku.
    <li>Odtwarzanie dwiku z filtrem Echo..
    <li>Zaptlanie dwiku.
    <li>Odtwarzanier jednoczenie maksymalnej liczby dwikw.
    <li>Pauza wszystkich dwikw.
    </ul>
    <p>Klasa ta nie jest zamieszczona w ksiace.
    @see SoundManager
    @see Sound
    @see SoundFilter
*/
public class SoundManagerTest extends GameCore
    implements ActionListener
{
    public static void main(String[] args) {
        new SoundManagerTest().run();
    }

    // nieskompresowany, 44100 Hz, 16-bitowy, mono, ze znakiem, 
    // rosncy porzdek bitw
    private static final AudioFormat PLAYBACK_FORMAT =
        new AudioFormat(44100, 16, 1, true, false);

    private static final int MANY_SOUNDS_COUNT =
        SoundManager.getMaxSimultaneousSounds(PLAYBACK_FORMAT);

    private static final int DRUM_TRACK = 1;

    private static final String EXIT = "Koniec";
    private static final String PAUSE = "Pauza";
    private static final String PLAY_MUSIC = "Odtwarzanie muzyki";
    private static final String MUSIC_DRUMS = "Przeczanie bbnw";
    private static final String PLAY_SOUND = "Odtwarzanie dwiku";
    private static final String PLAY_ECHO_SOUND = "Odtwarzanie dwiku z echem";
    private static final String PLAY_LOOPING_SOUND =
        "Odtwarzanie dwiku w ptli";
    private static final String PLAY_MANY_SOUNDS =
        "Odtwarzanie " + MANY_SOUNDS_COUNT + " dwikw";


    private SoundManager soundManager;
    private MidiPlayer midiPlayer;
    private Sequence music;
    private Sound boop;
    private Sound bzz;
    private InputStream lastloopingSound;

    public void init() {
        super.init();
        initSounds();
        initUI();
    }

    /**
        adowanie dwiku i muzyki.
    */
    public void initSounds() {
        midiPlayer = new MidiPlayer();
        soundManager = new SoundManager(PLAYBACK_FORMAT);
        music = midiPlayer.getSequence("../sounds/music.midi");
        boop = soundManager.getSound("../sounds/boop.wav");
        bzz = soundManager.getSound("../sounds/fly-bzz.wav");
    }


    /**
        Tworzenie interfejsu uytkownika, ktrym jest wiersz przyciskw.
    */
    public void initUI() {
        // Upewnij si, e komponenty Swing si nie rysuj
        NullRepaintManager.install();

        JFrame frame = super.screen.getFullScreenWindow();
        Container contentPane = frame.getContentPane();

        contentPane.setLayout(new FlowLayout());
        contentPane.add(createButton(PAUSE, true));
        contentPane.add(createButton(PLAY_MUSIC, true));
        contentPane.add(createButton(MUSIC_DRUMS, false));
        contentPane.add(createButton(PLAY_SOUND, false));
        contentPane.add(createButton(PLAY_ECHO_SOUND, false));
        contentPane.add(createButton(PLAY_LOOPING_SOUND, true));
        contentPane.add(createButton(PLAY_MANY_SOUNDS, false));
        contentPane.add(createButton(EXIT, false));

        // jawne ukadanie komponentw (potrzebne w niektrych systemach)
        frame.validate();
    }

    /**
        Rysowanie wszystkich komponentw Swing
    */
    public void draw(Graphics2D g) {
        JFrame frame = super.screen.getFullScreenWindow();
        frame.getLayeredPane().paintComponents(g);
    }

    /**
        Tworzenie przycisku (JButton lub JToggleButton).
    */
    public AbstractButton createButton(String name,
        boolean canToggle)
    {
        AbstractButton button;

        if (canToggle) {
            button = new JToggleButton(name);
        }
        else {
            button = new JButton(name);
        }
        button.addActionListener(this);
        button.setIgnoreRepaint(true);
        button.setFocusable(false);

        return button;
    }

    /**
        Wykonywanie akcji po naciniciu przycisku.
    */
    public void actionPerformed(ActionEvent e) {
        String command = e.getActionCommand();
        AbstractButton button = (AbstractButton)e.getSource();
        if (command == EXIT) {
            midiPlayer.close();
            soundManager.close();
            stop();
        }
        else if (command == PAUSE) {
            // pauzowanie dwiku
            soundManager.setPaused(button.isSelected());
            midiPlayer.setPaused(button.isSelected());
        }
        else if (command == PLAY_MUSIC) {
            // wczanie i wyczanie muzyki
            if (button.isSelected()) {
                midiPlayer.play(music, true);
            }
            else {
                midiPlayer.stop();
            }
        }
        else if (command == MUSIC_DRUMS) {
            // wczanie i wyczanie bbnw
            Sequencer sequencer = midiPlayer.getSequencer();
            if (sequencer != null) {
                 boolean mute = sequencer.getTrackMute(DRUM_TRACK);
                 sequencer.setTrackMute(DRUM_TRACK, !mute);
            }
        }
        else if (command == PLAY_SOUND) {
            // odtwarzanie normalnego dwiku
            soundManager.play(boop);
        }
        else if (command == PLAY_ECHO_SOUND) {
            // odtwarzanie dwiku z echem
            EchoFilter filter = new EchoFilter(11025, .6f);
            soundManager.play(boop, filter, false);
        }
        else if (command == PLAY_LOOPING_SOUND) {
            // odtwarzanie lub zatrzymywanie zaptlonego dwiku
            if (button.isSelected()) {
                lastloopingSound = soundManager.play(bzz, null, true);
            }
            else if (lastloopingSound != null) {
                try {
                    lastloopingSound.close();
                }
                catch (IOException ex) { }
                lastloopingSound = null;
            }
        }
        else if (command == PLAY_MANY_SOUNDS) {
            // Odtwarzanie jednoczenie kilku dwikw w celu przetestowania systemu
            for (int i=0; i<MANY_SOUNDS_COUNT; i++) {
                soundManager.play(boop);
            }
        }
    }

}
