package org.jpwh.test.filtering;

import org.hibernate.ReplicationMode;
import org.hibernate.Session;
import org.hibernate.criterion.MatchMode;
import org.hibernate.envers.AuditReader;
import org.hibernate.envers.AuditReaderFactory;
import org.hibernate.envers.DefaultRevisionEntity;
import org.hibernate.envers.RevisionType;
import org.hibernate.envers.query.AuditEntity;
import org.hibernate.envers.query.AuditQuery;
import org.jpwh.env.JPATest;
import org.jpwh.model.filtering.envers.Item;
import org.jpwh.model.filtering.envers.User;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;
import java.util.Date;
import java.util.List;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNull;

public class Envers extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("FilteringEnversPU");
    }

    @Test
    public void auditLogging() throws Throwable {

        UserTransaction tx = TM.getUserTransaction();
        try {

            Long ITEM_ID;
            Long USER_ID;
            {
                // Tworzenie
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                User user = new User("jandomanski");
                em.persist(user);

                Item item = new Item("Foo", user);
                em.persist(item);

                tx.commit();
                em.close();

                ITEM_ID = item.getId();
                USER_ID = user.getId();
            }
            Date TIMESTAMP_CREATE = new Date();

            {
                // Aktualizacja
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                Item item = em.find(Item.class, ITEM_ID);
                item.setName("Bar");
                item.getSeller().setUsername("domanskijan");

                tx.commit();
                em.close();
            }
            Date TIMESTAMP_UPDATE = new Date();

            {
                // Usuwanie
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                Item item = em.find(Item.class, ITEM_ID);
                em.remove(item);

                tx.commit();
                em.close();
            }
            Date TIMESTAMP_DELETE = new Date();

            {
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                /* 
                   Główny API Envers to klasa <code>AuditReader</code>. Do tego API można uzyskać dostę za pomocą
                   obiektu <code>EntityManager</code>.
                 */
                AuditReader auditReader = AuditReaderFactory.get(em);

                /* 
                   Na podstawie znacznika czasu możemy znaleźć numer wersji zbioru zmian
                   wprowadzonych przed datą odpowiadającą temu znacznikowi.
                 */
                Number revisionCreate = auditReader.getRevisionNumberForDate(TIMESTAMP_CREATE);
                Number revisionUpdate = auditReader.getRevisionNumberForDate(TIMESTAMP_UPDATE);
                Number revisionDelete  = auditReader.getRevisionNumberForDate(TIMESTAMP_DELETE);

                /* 
                    Jeśli nie mamy znacznika czasu, możemy pobrać wszystkie numery wersji, 
                    w których „uczestniczył” określony kontrolowany egzemplarz encji.
                    Ta operacja znajduje wszystkie zbiory zmian, w których określony 
                    obiekt <code>Item</code> był tworzony, modyfikowany, albo usunięty. 
                    W naszym przykładzie stworzyliśmy, zmodyfikowaliśmy, a następnie 
                    usunęliśmy egzemplarz <code>Item</code> Dlatego mamy trzy wersje.

                 */
                List<Number> itemRevisions = auditReader.getRevisions(Item.class, ITEM_ID);
                assertEquals(itemRevisions.size(), 3);
                for (Number itemRevision : itemRevisions) {
                    /* 
                          Jeśli dysponujemy numerem wersji, możemy pobrać znacznik czasu odpowiadający chwili, 
                          kiedy Envers zarejstrował zbiór zmian
                     */
                    Date itemRevisionTimestamp = auditReader.getRevisionDate(itemRevision);
                    // ...
                }

                /* 
                   Stworzyliśmy i zmodyfikowaliśmy obiekt <code>User</code>, dlatego mamy dwie wersje.
                 */
                List<Number> userRevisions = auditReader.getRevisions(User.class, USER_ID);
                assertEquals(userRevisions.size(), 2);

                em.clear();
                {
                    /* 
                       Jeśli znaczniki czasu, ani numery wersji  modyfikacji nie są znane, możemy napisać kwerendę z
                      <code>forRevisionsOfEntity()</code> w celu uzyskania wszystkich szczegółów inspekcji
                       określonej encji.
                     */
                    AuditQuery query = auditReader.createQuery()
                        .forRevisionsOfEntity(Item.class, false, false);

                    /* 
                       Ta kwerenda zwraca szczegóły dziennika inspekcji w postaci listy <code>List</code> tablic 
                       <code>Object[]</code>.
                     */
                    List<Object[]> result = query.getResultList();
                    for (Object[] tuple : result) {

                        /* 
                           Każda krotka wyników zawiera egzemplarz encji dla określonej wersji oraz
                           szczegóływ wersji (włącznie z numerem wersji i znacznikiem czasu), a także typem wersji.
                         */
                        Item item = (Item) tuple[0];
                        DefaultRevisionEntity revision = (DefaultRevisionEntity)tuple[1];
                        RevisionType revisionType = (RevisionType)tuple[2];

                        /* 
                           Typ wersji określa dlaczego Envers utworzył wersję , ponieważ egzemplarz encji
                           został wprowadzony, zmodyfikowany, albo usunięty z bazy danych.
                         */
                        if (revision.getId() == 1) {
                            assertEquals(revisionType, RevisionType.ADD);
                            assertEquals(item.getName(), "Foo");
                        } else if (revision.getId() == 2) {
                            assertEquals(revisionType, RevisionType.MOD);
                            assertEquals(item.getName(), "Bar");
                        } else if (revision.getId() == 3) {
                            assertEquals(revisionType, RevisionType.DEL);
                            assertNull(item);
                        }
                    }
                }
                em.clear();
                {
                    /* 
                       Metoda <code>find()</code> zwraca wersję śledzonego egzemplarza encji na podstawie 
                       poprawki. Ta operacja ładuje obiekt <code>Item</code> w postaci, w jakiej był po
                       utworzeniu.
                     */
                    Item item = auditReader.find(Item.class, ITEM_ID, revisionCreate);
                    assertEquals(item.getName(), "Foo");
                    assertEquals(item.getSeller().getUsername(), "jandomanski");

                    /* 
                       Ta operacja ładuje obiekt <code>Item</code> po jego aktualizacji. Zwróćmy uwagę, że
                       zmodyfikowan obiekt <code>seller</code> odpowiadający temu zbiorowi zmian także został
                       automatycznie pobrany.
                     */
                    Item modifiedItem = auditReader.find(Item.class, ITEM_ID, revisionUpdate);
                    assertEquals(modifiedItem.getName(), "Bar");
                    assertEquals(modifiedItem.getSeller().getUsername(), "domanskijan");

                    /* 
                       W tej rewizji obiekt <code>Item</code> został usunięty, dlatego metoda <code>find()</code>
                       zwróciła <code>null</code>.
                     */
                    Item deletedItem = auditReader.find(Item.class, ITEM_ID, revisionDelete);
                    assertNull(deletedItem);

                    /* 
                       W przykładzie nie zmodyfikowaliśmy jednak obiektu <code>User</code> w tej rewizji,
                       dlatego Envers zwraca najbliższą jej historyczną wersję .
                     */
                    User user = auditReader.find(User.class, USER_ID, revisionDelete);
                    assertEquals(user.getUsername(), "domanskijan");
                }
                em.clear();
                {
                    /* 
                       Ta kwerenda zwarca egzemplarze <code>Item</code> ograniczone do 
                       określonej rewizji i zbioru zmian.
                     */
                    AuditQuery query = auditReader.createQuery()
                        .forEntitiesAtRevision(Item.class, revisionUpdate);

                    /* 
                       Można dodać dodatkowe ograniczenia do kwerendy. Tutaj <code>Item#name</code>
                       musi zaczynać się od "Ba".
                     */
                    query.add(
                        AuditEntity.property("name").like("Ba", MatchMode.START)
                    );

                    /* 
                       Ograniczenia mogą obejmować asocjacje pomiędzy encjami. Na przykład szukamy
                       rewizji encji <code>Item</code> sprzedanej przez określonego użytkownika ( <code>User</code>).
                     */
                    query.add(
                        AuditEntity.relatedId("seller").eq(USER_ID)
                    );

                    /* 
                       Wyniki kwerendy mozna uporządkować.
                     */
                    query.addOrder(
                        AuditEntity.property("name").desc()
                    );

                    /* 
                       Obszerny zbiór wyników można podzielić na strony.
                     */
                    query.setFirstResult(0);
                    query.setMaxResults(10);

                    assertEquals(query.getResultList().size(), 1);
                    Item result = (Item)query.getResultList().get(0);
                    assertEquals(result.getSeller().getUsername(), "domanskijan");
                }
                em.clear();
                {
                    AuditQuery query = auditReader.createQuery()
                        .forEntitiesAtRevision(Item.class, revisionUpdate);

                    query.addProjection(
                        AuditEntity.property("name")
                    );

                    assertEquals(query.getResultList().size(), 1);
                    String result = (String)query.getSingleResult();
                    assertEquals(result, "Bar");
                }
                em.clear();
                {
                    // TODO To się nie powiedzie z kodem bajtowym instrumentacji. Bład  Hibernate  HHH-8600
                    /*
                    User user = auditReader.find(User.class, USER_ID, revisionCreate);

                    em.unwrap(Session.class)
                        .replicate(user, ReplicationMode.OVERWRITE);
                    em.flush();
                    em.clear();

                    user = em.find(User.class, USER_ID);
                    assertEquals(user.getUsername(), "jandomanski");
                    */
                }
                tx.commit();
                em.close();
            }
        } finally {
            TM.rollback();
        }
    }

}
