package org.jpwh.test.cache;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.stat.NaturalIdCacheStatistics;
import org.hibernate.stat.QueryStatistics;
import org.hibernate.stat.SecondLevelCacheStatistics;
import org.hibernate.stat.Statistics;
import org.jpwh.env.JPATest;
import org.jpwh.model.cache.Bid;
import org.jpwh.model.cache.Item;
import org.jpwh.model.cache.User;
import org.jpwh.shared.util.CalendarUtil;
import org.jpwh.shared.util.TestData;
import org.testng.annotations.Test;

import javax.management.ObjectName;
import javax.persistence.Cache;
import javax.persistence.CacheRetrieveMode;
import javax.persistence.CacheStoreMode;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Query;
import javax.transaction.UserTransaction;
import java.lang.management.ManagementFactory;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.testng.Assert.*;

public class SecondLevel extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("CachePU");
    }

    public static class CacheTestData {
        public TestData items;
        public TestData users;
    }

    public CacheTestData storeTestData() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();

        Long[] itemIds = new Long[3];
        Long[] userIds = new Long[3];

        User jandomanski = new User("jandomanski");
        em.persist(jandomanski);
        userIds[0] = jandomanski.getId();

        User janinadomanska = new User("janinadomanska");
        em.persist(janinadomanska);
        userIds[1] = janinadomanska.getId();

        User robertdomanski = new User("robertdomanski");
        em.persist(robertdomanski);
        userIds[2] = robertdomanski.getId();

        Item item = new Item("Przedmiot pierwszy", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[0] = item.getId();
        for (int i = 1; i <= 3; i++) {
            Bid bid = new Bid(item, robertdomanski, new BigDecimal(9 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot drugi", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[1] = item.getId();
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, janinadomanska, new BigDecimal(2 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot_trzeci", CalendarUtil.AFTER_TOMORROW.getTime(), janinadomanska);
        em.persist(item);
        itemIds[2] = item.getId();
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, jandomanski, new BigDecimal(3 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        tx.commit();
        em.close();

        // Prawidłowo "rozgrzej" pamięć podręczną
        tx.begin();
        em = JPA.createEntityManager();

        //Tutaj ładujemy wszystkie egzemplarze User do pamięci podręcznej drugiego poziomu, 
		//ponieważ strategia NONSTRICT_READ_WRITE nie wstawi ich do pamięci podręcznej 
		//podczas wywołania persist(), a dopiero podczas ich ładowania z bazy danych. 
		//Trzeba to zrobić w nowej transakcji. 
		//Jeśli się tego nie zrobi, to operacja TwoPhaseLoad nie załaduje danych do pamięci podręcznej.
		em.createQuery("select u from User u").getResultList();

        // Elementy kolekcji Item#bids są ładowane do pamięci podrtęcznej w momencie
        // ładowania kolekcji.
        for (Long itemId : itemIds) {
            em.find(Item.class, itemId).getBids().size();
        }
        tx.commit();
        em.close();

        JPA.getEntityManagerFactory()
            .unwrap(SessionFactory.class)
            .getStatistics().clear();

        CacheTestData testData = new CacheTestData();
        testData.items = new TestData(itemIds);
        testData.users = new TestData(userIds);
        return testData;
    }

    @Test
    public void cacheBehavior() throws Exception {
        CacheTestData testData = storeTestData();
        Long USER_ID = testData.users.getFirstId();
        Long ITEM_ID = testData.items.getFirstId();

        UserTransaction tx = TM.getUserTransaction();
        try {

            {
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                // Pobranie API statystyk
                Statistics stats =
                    JPA.getEntityManagerFactory()
                        .unwrap(SessionFactory.class)
                        .getStatistics();

                SecondLevelCacheStatistics itemCacheStats =
                    stats.getSecondLevelCacheStatistics(Item.class.getName());
                assertEquals(itemCacheStats.getElementCountInMemory(), 3);
                assertEquals(itemCacheStats.getHitCount(), 0);

                // Dostęp do pamięci podręcznej drugiego poziomu za pomocą mechanizmu wyszukiwania encji według identyfikatora
                Item item = em.find(Item.class, ITEM_ID);
                assertEquals(itemCacheStats.getHitCount(), 1);

                // Inicjalizacja proxy także spowoduje sięgnięcie do pamięci podrtęcznej drugiego poziomu
                SecondLevelCacheStatistics userCacheStats =
                    stats.getSecondLevelCacheStatistics(User.class.getName());
                assertEquals(userCacheStats.getElementCountInMemory(), 3);
                assertEquals(userCacheStats.getHitCount(), 0);

                User seller = item.getSeller();
                assertEquals(seller.getUsername(), "jandomanski"); // Inicjalizacja proxy

                assertEquals(userCacheStats.getHitCount(), 1);

                // Pobranie kolekcji Item#bids i egzemplarzy encji Bid, do których się odwołuje
                /* 
                   Ze statystyk wiemy, że mamy trzy elementy kolekcji <code>Item#bids</code>
                   w pamięci podręcznej (po jednej dla każdego obiektu <code>Item</code>). Do tej pory
                   nie nastąpiły zakończone sukcesem wyszukiwania w pamięci podręcznej.
                 */
                SecondLevelCacheStatistics bidsCacheStats =
                    stats.getSecondLevelCacheStatistics(Item.class.getName() + ".bids");
                assertEquals(bidsCacheStats.getElementCountInMemory(), 3);
                assertEquals(bidsCacheStats.getHitCount(), 0);

                /* 
                   Pamięć podręczna encji dla encji <code>Bid</code> zawiera pięć rekordów. Nie 
                   sięgaliśmy jeszcze do nich.
                 */
                SecondLevelCacheStatistics bidCacheStats =
                    stats.getSecondLevelCacheStatistics(Bid.class.getName());
                assertEquals(bidCacheStats.getElementCountInMemory(), 5);
                assertEquals(bidCacheStats.getHitCount(), 0);

                /* 
                   Inicjalizacja kolekcji spowoduje odczytanie danych z obu pamięci podręcznych.
                 */
                Set<Bid> bids = item.getBids();
                assertEquals(bids.size(), 3);

                /* 
                   W pamięci podręcznej znaleziono jedną kolekcję oraz dane dla
                   jej trzech elementów <code>Bid</code>.
                 */
                assertEquals(bidsCacheStats.getHitCount(), 1);
                assertEquals(bidCacheStats.getHitCount(), 3);

                tx.commit();
                em.close();
            }

        } finally {
            TM.rollback();
        }
    }

    @Test
    public void cacheModes() throws Exception {
        CacheTestData testData = storeTestData();
        Long USER_ID = testData.users.getFirstId();
        Long ITEM_ID = testData.items.getFirstId();

        UserTransaction tx = TM.getUserTransaction();
        try {

            {
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                Statistics stats =
                    JPA.getEntityManagerFactory()
                        .unwrap(SessionFactory.class)
                        .getStatistics();

                SecondLevelCacheStatistics itemCacheStats =
                    stats.getSecondLevelCacheStatistics(Item.class.getName());

                // Pominięcie pamięci podręcznej w momencie pobierania egzemplarza encji według identyfikatora
                {
                    Map<String, Object> properties = new HashMap<String, Object>();
                    properties.put("javax.persistence.cache.retrieveMode", CacheRetrieveMode.BYPASS);
                    Item item = em.find(Item.class, ITEM_ID, properties); // Dostęp do bazy danych
                    assertEquals(itemCacheStats.getHitCount(), 0);
                }

                // Pominięcie pamięci podręcznej w momencie zapisywania egzemplarza encji
                assertEquals(itemCacheStats.getElementCountInMemory(), 3);
                em.setProperty("javax.persistence.cache.storeMode", CacheStoreMode.BYPASS);

                Item item = new Item(
                    // ...
                    "Jakiś przedmiot",
                    CalendarUtil.TOMORROW.getTime(),
                    em.find(User.class, USER_ID)
                );

                em.persist(item); // Nie jest zapisany w pamięci podręcznej

                em.flush();
                assertEquals(itemCacheStats.getElementCountInMemory(), 3); // Bez zmian

                tx.commit();
                em.close();
            }

        } finally {
            TM.rollback();
        }
    }

    @Test
    public void cacheNaturalId() throws Exception {
        CacheTestData testData = storeTestData();
        Long USER_ID = testData.users.getFirstId();
        Long ITEM_ID = testData.items.getFirstId();

        UserTransaction tx = TM.getUserTransaction();
        try {

            Statistics stats =
                JPA.getEntityManagerFactory()
                    .unwrap(SessionFactory.class)
                    .getStatistics();

            // Wyczyszczenie wszystkich naturalnych regionów pamięci podręcznej ID
            JPA.getEntityManagerFactory().getCache()
                .unwrap(org.hibernate.Cache.class)
                .evictNaturalIdRegions();

            // Wyczyszczenie obszaru pamięci podręcznej encji User
            JPA.getEntityManagerFactory().getCache().evict(User.class);

            {
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                Session session = em.unwrap(Session.class);

                NaturalIdCacheStatistics userIdStats =
                    stats.getNaturalIdCacheStatistics(User.class.getName() + "##NaturalId");

                assertEquals(userIdStats.getElementCountInMemory(), 0);

                User user = (User) session.byNaturalId(User.class)
                    .using("username", "jandomanski")
                    .load();
                // select ID from USERS where USERNAME = ?
                // select * from USERS where ID = ?

                assertNotNull(user);

                assertEquals(userIdStats.getHitCount(), 0);
                assertEquals(userIdStats.getMissCount(), 1);
                assertEquals(userIdStats.getElementCountInMemory(), 1);

                SecondLevelCacheStatistics userStats =
                    stats.getSecondLevelCacheStatistics(User.class.getName());
                assertEquals(userStats.getHitCount(), 0);
                assertEquals(userStats.getMissCount(), 1);
                assertEquals(userStats.getElementCountInMemory(), 1);

                tx.commit();
                em.close();
            }

            { // Ponowne uruchomienie wyszukiwania. Sięgnięcie do pamięci podręcznej
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                Session session = em.unwrap(Session.class);

                /* 
                   Region pamięci podręcznej naturalnego identyfikatora dla <code>User</code>
                   zawiera jeden element.
                 */
                NaturalIdCacheStatistics userIdStats =
                    stats.getNaturalIdCacheStatistics(User.class.getName() + "##NaturalId");
                assertEquals(userIdStats.getElementCountInMemory(), 1);

                /* 
                   API <code>org.hibernate.Session</code> realizuje wyszukiwanie
                   naturalnego identyfikatora. To jest jedyny API dostępu do
                   pamięci podręcznej naturalnego identyfikatora.
                 */
                User user = (User) session.byNaturalId(User.class)
                    .using("username", "jandomanski")
                    .load();

                assertNotNull(user);

                /* 
                   Trafiliśmy w pamięć podręczną podczas wyszukiwania naturalnego identyfikatora. 
                   Pamięć podręczna zwróciła nową wartosć identyfikatora "jandomanski".
                 */
                assertEquals(userIdStats.getHitCount(), 1);

                /* 
                   Dla właściwych danych encji 
                   <code>User</code> także trafiliśmy w pamięć podręczną.
                 */
                SecondLevelCacheStatistics userStats =
                    stats.getSecondLevelCacheStatistics(User.class.getName());
                assertEquals(userStats.getHitCount(), 1);

                tx.commit();
                em.close();
            }

        } finally {
            TM.rollback();
        }
    }

    @Test
    public void cacheControl() throws Exception {
        CacheTestData testData = storeTestData();
        Long USER_ID = testData.users.getFirstId();
        Long ITEM_ID = testData.items.getFirstId();

        EntityManagerFactory emf = JPA.getEntityManagerFactory();
        Cache cache = emf.getCache();

        assertTrue(cache.contains(Item.class, ITEM_ID));
        cache.evict(Item.class, ITEM_ID);
        cache.evict(Item.class);
        cache.evictAll();

        org.hibernate.Cache hibernateCache =
            cache.unwrap(org.hibernate.Cache.class);

        assertFalse(hibernateCache.containsEntity(Item.class, ITEM_ID));
        hibernateCache.evictEntityRegions();
        hibernateCache.evictCollectionRegions();
        hibernateCache.evictNaturalIdRegions();
        hibernateCache.evictQueryRegions();
    }

    @Test
    public void cacheQueryResults() throws Exception {
        CacheTestData testData = storeTestData();
        Long USER_ID = testData.users.getFirstId();
        Long ITEM_ID = testData.items.getFirstId();

        UserTransaction tx = TM.getUserTransaction();
        try {

            // Wyczyszczenie obszaru pamięci podręcznej
            JPA.getEntityManagerFactory().getCache().evict(Item.class);

            Statistics stats =
                JPA.getEntityManagerFactory()
                    .unwrap(SessionFactory.class)
                    .getStatistics();

            {
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                String queryString = "select i from Item i where i.name like :n";

                /* 
                   Trzeba włączyć buforowanie dla konkretnego zapytania. Bez wskazówki
                   <code>org.hibernate.cachable</code>, wynik 
                   nie zostałby zapisany w pamięci podręcznej wyników zapytania.
                 */
                Query query = em.createQuery(queryString)
                    .setParameter("n", "I%")
                    .setHint("org.hibernate.cacheable", true);

                /* 
                   Hibernate uruchomi teraz zapytanie SQL i pobierze
                   wyniki do pamięci.                 */
                List<Item> items = query.getResultList();
                assertEquals(items.size(), 3);

                /* 
                   Używanie API statystyk pozwala uzyskać więcej szczegółów.
                   To jest pierwszy raz, gdy uruchamiamy to zapytania, dlatego
                   było chybienie, a nie trafienie w pamięć podręczną. Hibernate umieszcza zapytanie oraz
                   jego wynik w pamięci podręcznej. Jeśli jeszcze raz uruchomimy dokładnie to samo zapytanie,
                   to wynik będzie pobrany z pamięci podręcznej.
                 */
                QueryStatistics queryStats = stats.getQueryStatistics(queryString);
                assertEquals(queryStats.getCacheHitCount(), 0);
                assertEquals(queryStats.getCacheMissCount(), 1);
                assertEquals(queryStats.getCachePutCount(), 1);

                /* 
                   Właściwy egzemplarz encji pobrany w zestawie wyników 
                   jest zapisany w regionie pamięci podręcznej encji, a nie w pamięci podręcznej wyników zapytania.
                 */
                SecondLevelCacheStatistics itemCacheStats =
                    stats.getSecondLevelCacheStatistics(Item.class.getName());
                assertEquals(itemCacheStats.getElementCountInMemory(), 3);

                tx.commit();
                em.close();
            }

            { // Ponowne uruchomienie zapytania. Trafienie w pamięć podręczną
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                String queryString = "select i from Item i where i.name like :n";

                List<Item> items = em.createQuery(queryString)
                    .setParameter("n", "I%")
                    .setHint("org.hibernate.cacheable", true)
                    .getResultList();

                assertEquals(items.size(), 3);

                QueryStatistics queryStats = stats.getQueryStatistics(queryString);
                assertEquals(queryStats.getCacheHitCount(), 1);
                assertEquals(queryStats.getCacheMissCount(), 1);
                assertEquals(queryStats.getCachePutCount(), 1);

                tx.commit();
                em.close();
            }

        } finally {
            TM.rollback();
        }
    }

    @javax.management.MXBean
    public interface StatisticsMXBean extends Statistics {
    }

    public void exposeStatistics(final Statistics statistics) throws Exception {
        statistics.setStatisticsEnabled(true);
        Object statisticsBean = Proxy.newProxyInstance(
            getClass().getClassLoader(), new Class<?>[]{StatisticsMXBean.class}, new InvocationHandler() {
                @Override
                public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
                    return method.invoke(statistics, args);
                }
            });
        ManagementFactory.getPlatformMBeanServer()
            .registerMBean(
                statisticsBean,
                new ObjectName("org.hibernate:type=statistics")
            );
    }
}
