package org.jpwh.test.bulkbatch;

import org.jpwh.env.JPATest;
import org.jpwh.model.bulkbatch.Item;
import org.jpwh.model.bulkbatch.User;
import org.jpwh.shared.util.CalendarUtil;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;
import java.util.Date;
import java.util.logging.Logger;

import static org.testng.Assert.assertEquals;

public class BatchInsertUpdate extends JPATest {

    final private static Logger log = Logger.getLogger(BatchInsertUpdate.class.getName());

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("BulkBatchPU");
    }

    @Test
    public void batchInsertUpdate() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        try {
            long ONE_HUNDRED_THOUSAND = 10000; // Tak, kłamiemy tutaj, aby test nie działał przez 5 minut
            {
                tx.setTransactionTimeout(300); // 5 minut, to jest API UserTransaction
                //Tylko przyszłe transakcje uruchomione w tym wątku będą miały nowy timeout.

                long startTime = new Date().getTime();
                tx.begin();
                EntityManager em = JPA.createEntityManager();

                User jandomanski = new User("jandomanski");
                em.persist(jandomanski);

                /* 
                   Tutaj tworzymy i utrwalanmmy 100 000 egzemplarzy <code>Item</code>.
                 */
                for (int i = 0; i < ONE_HUNDRED_THOUSAND; i++) {
                    Item item = new Item(
                        // ...
                       "Item " + i, CalendarUtil.TOMORROW.getTime(), jandomanski
                    );
                    em.persist(item);

                    /* 
                       Po 100 operacjach, wykonujemy flush i czyścimy kontekst utrwalania. Powoduje to uruchomienie 
                       instrukcji SQL <code>INSERT</code> dla 100 egzemplarzy <code>Item</code>, a ponieważ
                       teraz są one w stanie odłączonym i nie są już wykorzystywane, to mechanizm odśmiecania JVM
                       może odzyskać pamięć.
                     */
                    if (i % 100 == 0) {
                        em.flush();
                        em.clear();
                    }
                }

                tx.commit();
                em.close();

                long endTime = new Date().getTime();
                log.info("### Czas wstawiania paczki w sekundach: " + ((endTime-startTime)/1000));
            }
            {
                // Sprawdzenie, czy wszystkie przedmioty zostały wprowadzone
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                assertEquals(
                   em.createQuery("select count(i) from Item i").getSingleResult(),
                   ONE_HUNDRED_THOUSAND
                );
                tx.commit();
                em.close();
            }
            {
                long startTime = new Date().getTime();

                tx.begin();
                EntityManager em = JPA.createEntityManager();

                /* 
                   Używamy zapytania JPQL w celu załadowania wszystkich egzemplarzy <code>Item</code> z
                   bazy danych. Zamiast pobierania wyniku zapytania w całości do 
                   pamięci aplikacji, otwieramy kursor bazy danych online.
                 */
                org.hibernate.ScrollableResults itemCursor =
                   em.unwrap(org.hibernate.Session.class)
                       .createQuery("select i from Item i")
                       .scroll(org.hibernate.ScrollMode.SCROLL_INSENSITIVE);

                int count = 0;
                /* 
                    Zarządzamy kursorem za pomocą API <code>ScrollableResults</code> i przesuwamy się po
                    wynikach. Każde wywołanie metody <code>next()</code> przesuwa kursor wprzód do następnego rekordu.
                 */
                while (itemCursor.next()) {
                    /* 
                       Wywołanie <code>get(int i)</code> pobiera do pamięci pojedynczy egzemplarz encji,
                       rekord, na który aktualnie wskazuje kursor.
                     */
                    Item item = (Item) itemCursor.get(0);

                    modifyItem(item);

                    /* 
                       Aby uniknąć wyczerpania pamięci, wykonujemy operację flush i czyścimy kontekst utrwalania
                       przed załadowaniem do niego kolejnych 100 rekordów.
                     */
                    if (++count % 100 == 0) { // Ustawienie hibernate.jdbc.batch_size na 100!
                        em.flush();
                        em.clear();
                    }
                }

                itemCursor.close();
                tx.commit();
                em.close();

                long endTime = new Date().getTime();
                log.info("### Czas aktualizowania paczki w sekundach: " + ((endTime-startTime)/1000));
            }
            {
                // Sprawdzenie, czy wszystkie przedmioty zostały zaktualizowane
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                assertEquals(
                   em.createQuery("select count(i) from Item i where i.active = false").getSingleResult(),
                   0l
                );
                tx.commit();
                em.close();
            }

        } finally {
            TM.rollback();
        }
    }

    // TODO: Błędy
    @Test
    public void batchInsertUpdateWithStatelessSession() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        try {

            long ONE_HUNDRED_THOUSAND = 10000; // Tak, kłamiemy tutaj, aby test nie działał przez 5 minut
            {
                tx.begin();
                org.hibernate.SessionFactory sf =
                    JPA.getEntityManagerFactory().unwrap(org.hibernate.SessionFactory.class);
                org.hibernate.StatelessSession statelessSession = sf .openStatelessSession();

                User jandomanski = new User("jandomanski");
                statelessSession.insert(jandomanski);

                for (int i = 0; i < ONE_HUNDRED_THOUSAND; i++) {
                    Item item = new Item(
                       "Item " + i, CalendarUtil.TOMORROW.getTime(), jandomanski
                    );

                    statelessSession.insert(item);
                }

                tx.commit();
                statelessSession.close();
            }
            {
                tx.begin();
                org.hibernate.SessionFactory sf =
                    JPA.getEntityManagerFactory().unwrap(org.hibernate.SessionFactory.class);
                org.hibernate.StatelessSession statelessSession = sf .openStatelessSession();

                // TODO: Powinienem napotkać na poniższy błąd, ale wszystko działa: https://hibernate.atlassian.net/browse/HHH-4042
                long count = (Long)statelessSession.createQuery("select count(i) from Item i").uniqueResult();
                assertEquals(count, ONE_HUNDRED_THOUSAND);

                tx.commit();
                statelessSession.close();
            }
            {
                long startTime = new Date().getTime();

                // TODO: To nie powiedzie się, jeśli ustalimy rozmiar paczki w User.java

                tx.begin();

                /* 
                   Otwieramy sesję <code>StatelessSession</code> za pomocą fabryki <code>SessionFactory</code> frameworka Hibernate.
                   Możemy ją odpakowa z <code>EntityManagerFactory</code>.
                 */
                org.hibernate.SessionFactory sf =
                    JPA.getEntityManagerFactory().unwrap(org.hibernate.SessionFactory.class);
                org.hibernate.StatelessSession statelessSession = sf .openStatelessSession();

                /* 
                   Używamy zapytania JPQL w celu załadowania wszystkich egzemplarzy <code>Item</code> z
                   bazy danych. Zamiast pobierania wyniku zapytania w całości do 
                   pamięci aplikacji, otwieramy kursor bazy danych online.
                 */
                org.hibernate.ScrollableResults itemCursor =
                    statelessSession
                        .createQuery("select i from Item i")
                       .scroll(org.hibernate.ScrollMode.SCROLL_INSENSITIVE);

                /* 
                   Przemieszczamy się po wyniku z wykorzystaniem kursora i pobieramy egzemplarz encji <code>Item</code>.
                   Ten egzemplarz jest w stanie odłączonym. Nie ma kontekstu utrwalania!
                 */
                while (itemCursor.next()) {
                    Item item = (Item) itemCursor.get(0);

                    modifyItem(item);

                    /* 
                       Ponieważ Hibertnate bez kontekstu utrwalania nie wykrywa automnatycznie zmian, to
                       trzeba ręcznie uruchomić instrukcje SQL <code>UPDATE</code>.
                     */
                    statelessSession.update(item);
                }

                itemCursor.close();
                tx.commit();
                statelessSession.close();

                long endTime = new Date().getTime();
                log.info("### Czas aktualizowania sesji stateless w sekundach: " + ((endTime - startTime) / 1000));
            }
            {
                // Sprawdzenie, czy wszystkie przedmioty zostały zaktualizowane
                tx.begin();
                EntityManager em = JPA.createEntityManager();
                assertEquals(
                   em.createQuery("select count(i) from Item i where i.active = false").getSingleResult(),
                   0l
                );
                tx.commit();
                em.close();
            }

        } finally {
            TM.rollback();
        }
    }
    protected void modifyItem(Item item) {
        item.setActive(true); // To jest trywialne, ale można zobaczyć, o co chodzi
    }

}

