package org.jpwh.helloworld;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.boot.Metadata;
import org.hibernate.boot.MetadataBuilder;
import org.hibernate.boot.MetadataSources;
import org.hibernate.boot.registry.StandardServiceRegistryBuilder;
import org.hibernate.cfg.Environment;
import org.hibernate.resource.transaction.backend.jta.internal.JtaTransactionCoordinatorBuilderImpl;
import org.hibernate.service.ServiceRegistry;
import org.jpwh.env.TransactionManagerTest;
import org.jpwh.model.helloworld.Message;
import org.testng.annotations.Test;

import javax.transaction.UserTransaction;
import java.util.List;

import static org.testng.Assert.*;

public class HelloWorldHibernate extends TransactionManagerTest {

    protected void unusedSimpleBoot() {
        SessionFactory sessionFactory = new MetadataSources(
            new StandardServiceRegistryBuilder()
                .configure("hibernate.cfg.xml").build()
        ).buildMetadata().buildSessionFactory();
    }

    protected SessionFactory createSessionFactory() {

        /* 
            Ten budowniczy pomaga utworzyć niemutowalny rejestr usług z wykorzystaniem łańcucha wywołań metod.
         */
        StandardServiceRegistryBuilder serviceRegistryBuilder =
            new StandardServiceRegistryBuilder();

        /* 
            Konfiguracja rejestru usług poprzez zastosowanie ustawień.
         */
        serviceRegistryBuilder
            .applySetting("hibernate.connection.datasource", "myDS")
            .applySetting("hibernate.format_sql", "true")
            .applySetting("hibernate.use_sql_comments", "true")
            .applySetting("hibernate.hbm2ddl.auto", "create-drop");

        // Włączenie JTA (to jest trochę prymitywne, ponieważ deweloperzy Hibernate nadal sądzą, że JTA
        // jest używane tylko w monstrualnych serwerach aplikacji, i że nigdy nie zobaczymy tego kodu).
        serviceRegistryBuilder.applySetting(
            Environment.TRANSACTION_COORDINATOR_STRATEGY,
            JtaTransactionCoordinatorBuilderImpl.class
        );
        ServiceRegistry serviceRegistry = serviceRegistryBuilder.build();

        /* 
            Do teh fazy konfiguracji możemy się dostać tylko z istniejącym rejestrem usług.
         */
        MetadataSources metadataSources = new MetadataSources(serviceRegistry);

        /* 
            Dodanie klas utrwalania do (mapowania) źródeł metadanych.
         */
        metadataSources.addAnnotatedClass(
            org.jpwh.model.helloworld.Message.class
        );

        // Dodanie plików mapowania hbm.xml
        // metadataSources.addFile(...);

        // Odczytanie wszystkich plików mapowania hbm.xml z pliku JAR
        // metadataSources.addJar(...)

        MetadataBuilder metadataBuilder = metadataSources.getMetadataBuilder();

        Metadata metadata = metadataBuilder.build();

        assertEquals(metadata.getEntityBindings().size(), 1);

        SessionFactory sessionFactory = metadata.buildSessionFactory();

        return sessionFactory;
    }

    @Test
    public void storeLoadMessage() throws Exception {
        SessionFactory sessionFactory = createSessionFactory();
        try {
            {
                /* 
                    Uzyskanie dostępu do standardowego API transakcji <code>UserTransaction</code> i
                    rozpoczęcie transakcji dla tego wątku wykonania
                 */
                UserTransaction tx = TM.getUserTransaction();
                tx.begin();

                /* 
                    Zawsze, kiedy wywołujesz <code>getCurrentSession()</code> w tym samym wątku, otrzymasz
                    ten sam <code>org.hibernate.Session</code>. Jest on powiązany automatycznie z
                    bieżącą transakcją i jest zamykany automatycznie, kiedy ta transakcja zostanie
                    zatwierdzona, albo wycofana
                 */
                Session session = sessionFactory.getCurrentSession();

                Message message = new Message();
                message.setText("Witaj świecie!");

                /* 
                    Natywny interfejs API frameworka Hibernate jest bardzo podobny do standardowego Java Persistence API. Większość metod ma te same nazwy.
                 */
                session.persist(message);

                /* 
                    Hibernate synchronizuje sesję z bazą danych i automatycznie zamyka „bieżącą” sesję w momencie zatwierdzenia powiązanej transakcji.

                 */
                tx.commit();
                // INSERT into MESSAGE (ID, TEXT) values (1, 'Witaj świecie!')
            }

            {
                UserTransaction tx = TM.getUserTransaction();
                tx.begin();

                /* 
                    Zapytanie kryteriów frameworka Hibernate to zapewniający bezpieczeństwo typów programowy sposób wyrażania zapytań automatycznie tłumaczonych na SQL.
                 */
                List<Message> messages =
                    sessionFactory.getCurrentSession().createCriteria(
                        Message.class
                    ).list();
                // SELECT * from MESSAGE

                assertEquals(messages.size(), 1);
                assertEquals(messages.get(0).getText(), "Witaj świecie!");

                tx.commit();
            }

        } finally {
            TM.rollback();
        }
    }
}

