package org.jpwh.web.jaxrs;

import org.jpwh.web.dao.ItemDAO;
import org.jpwh.web.model.Image;
import org.jpwh.web.model.Item;

import javax.inject.Inject;
import javax.transaction.Transactional;
import javax.ws.rs.*;
import java.util.List;

import static javax.ws.rs.core.MediaType.APPLICATION_XML;
import static javax.ws.rs.core.Response.Status.NOT_FOUND;

/* 
	
	Gdy serwer otrzyma żądanie ze ścieżką /item, to jest ono obsługiwane przez metodę tej usługi. Domyślnie egzemplarz 
	usługi ma zasięg żądania, ale można to zmienić poprzez zastosowanie adnotacji CDI służących do definiowania zasięgu.
 */
@Path("/item")
public class ItemService {

   ItemDAO itemDAO;

    /*         
		Metoda ta jest mapowana na żądanie HTTP GET
     */
    @GET
    /* 
    
		W roli wartości argumentu dla wywołania kontener wykorzystuje 
		segment ścieżki za fragmentem /item, na przykład /item/123. 
		Tę wartość mapujemy na parametr metody za pomocą adnotacji @PathParam.
		
      */
    @Path("{id}")
    /* 
        
		Ta metoda generuje medium XML, dlatego jakiś mechanizm musi 
		zserializować wartość zwracaną przez metodę do formatu XML. 
		Należy jednak zachować ostrożność: ta adnotacja nie jest 
		tą samą adnotacją producenta, jak w CDI. To jest inny pakiet!
		
     */
    @Produces(APPLICATION_XML)
    public Item get(@PathParam("id") Long id) {
        Item item = itemDAO.findById(id);
        if (item == null)
            throw new WebApplicationException(NOT_FOUND);
        return item;
    }

    @PUT
    @Path("{id}")
    /* 
		
		Metoda przetwarza medium XML, dlatego jakiś mechanizm musi 
		zdeserializować dokument XML, i przekształcić go na egzemplarz 
		Item w stanie odłączonym.
		
    */
    @Consumes(APPLICATION_XML)
    /* 
        
		W tej metodzie chcemy zapisać dane. W tym celu trzeba rozpocząć 
		transakcję systemową i połączyć z nią kontekst utrwalania.
    */
    @Transactional
    public void put(@PathParam("id") Long id, Item item) {
        itemDAO.joinTransaction();
        itemDAO.makePersistent(item);
    }

    @GET
    @Path("{id}/images")
    @Produces({APPLICATION_XML})
    @Transactional
    public List<Image> getImages(@PathParam("id") Long id) {
        Item item = itemDAO.findById(id);
        if (item == null)
            throw new WebApplicationException(NOT_FOUND);
        return item.getImagesSorted();
    }
}