package org.jpwh.web.jaxrs;

import org.hibernate.proxy.HibernateProxy;
import org.hibernate.proxy.HibernateProxyHelper;
import org.hibernate.proxy.LazyInitializer;
import org.jpwh.web.model.EntityReference;

import javax.persistence.EntityManager;
import javax.xml.bind.annotation.adapters.XmlAdapter;
import java.lang.reflect.Field;

public class EntityReferenceAdapter
    extends XmlAdapter<EntityReference, Object> {

    EntityManager em;

    /* 
		JAXB wywołuje ten konstruktor w momencie, kiedy generuje dokument XML. 
		W tym przypadku nie potrzebujemy obiektu EntityManager: obiekt proxy 
		zawiera wszystkie informacje potrzebne do zapisania obiektu EntityReference.
     */
    public EntityReferenceAdapter() {
    }

    /* 
        JAXB must call this constructor when it reads an XML document.
        You need an <code>EntityManager</code> to get
        a Hibernate proxy from an <code>EntityReference</code>.
		
		JAXB musi wywołać ten konstruktor w momencie odczytania dokumentu XML. 
		<code>EntityManager</code> jest potrzebny do odczytania obiektu proxy z 
		<code>EntityReference</code>.
		
     */
    public EntityReferenceAdapter(EntityManager em) {
        this.em = em;
    }

    @Override
    public EntityReference marshal(Object entityInstance)
        throws Exception {

        /* 
            
			Podczas pisania dokumentu XML, należy pobrać proxy Hibernate, 
			i utworzyć reprezentację, która implementuje interfejs serializable. 
			To wywołuje wewnętrzne metody Hibernate, których tutaj nie pokazaliśmy.
			
         */
        Class type = getType(entityInstance);
        Long id = getId(type, entityInstance);
        return new EntityReference(type, id);
    }

    @Override
    public Object unmarshal(EntityReference entityReference)
        throws Exception {
        if (em == null)
            throw new IllegalStateException(
                "Wywołaj Unmarshaller#setAdapter() i " +
                    "dostarcz obiekt EntityManager"
            );

        /* 
           
		   Podczas czytania dokumentu XML pobierz zserializowaną reprezentację, 
		   i stwórz obiekt proxy frameworka Hibernate dołączony do aktualnego 
		   kontekstu utrwalania.
		   
         */
        return em.getReference(
            entityReference.type,
            entityReference.id
        );
    }

    // To są zastrzeżone operacje Hibernate. Zakładamy, że encje JPA
	// są mapowane z dostępem do pól (sprawdzanie pola 'id'). Z łatwością można
	// przepisać ten kod tak, aby korzystał z metod gettera/settera dla identyfikatora

    protected Class getType(Object o) throws Exception {
        return HibernateProxyHelper.getClassWithoutInitializingProxy(o);
    }

    protected Long getId(Class type, Object entityInstance) throws Exception {
        if (entityInstance instanceof HibernateProxy) {
            LazyInitializer lazyInitializer =
                ((HibernateProxy) entityInstance).getHibernateLazyInitializer();
            return (Long) lazyInitializer.getIdentifier();
        }
        return (Long) getIdField(type).get(entityInstance);
    }

    protected Field getIdField(Class type) throws Exception {
        Field idField = type.getDeclaredField("id");
        if (idField == null || idField.getType() != Long.class)
            throw new IllegalArgumentException("Brak pola 'id' typu Long w: " + type);
        return idField;
    }
}
