    public AuctionClient() throws Exception {
        System.out.println(">>> Rozpoczęcie dialogu, nawiązanie połączenia z serwerem (wciśnij CTRL+C, aby zakończyć)...");
        this.auctionService = lookupService(RemoteAuctionService.class, AuctionServiceImpl.class);
    }

    @SuppressWarnings("InfiniteRecursion")
    public void startDialog() throws Exception {

        // Pobranie listy przedmiotów z usługi
        List<Item> items = auctionService.getItems(true);

        // Wyrenderowanie listy
        renderItemBidSummary(items);

        // Umożliwienie użytkownikowi na pobranie jednego przedmiotu (stan odłączony) i akcji do wykonania
        Item item = promptItem(items);
        System.out.println("Czy chcesz zmienić nazwę (n) przedmiot, czy złożyć ofertę (b):");
        String action = userInput.readLine();
        if (action.equals("n")) {
            changeName(item);
        } else if (action.equals("b")) {
            placeBid(item);
        }

        startDialog();
    }

    protected void renderItemBidSummary(List<Item> items) {
        if (items.size() == 0) {
            System.err.println("Nie znaleziono przedmiotów. Czy na pewno zaimportowałeś dane testowe? Kończę pracę...");
            System.exit(1);
        }
        System.out.println("--------------------------------------------------------------------");
        String format = "%-4s | %-20s | %-20s | %15s %n";
        System.out.printf(format, "ID", "Nazwa", "Koniec aukcji", "Najwyższa oferta");
        System.out.println("--------------------------------------------------------------------");
        for (Item item : items) {
            Bid highestBid = item.getHighestBid();
            String highestBidAmount = highestBid != null ? highestBid.getAmount().toString() : "-";
            String auctionEnd = dateFormat.format(item.getAuctionEnd());
            System.out.printf(format, item.getId(), item.getName(), auctionEnd, highestBidAmount);
        }
        System.out.println("--------------------------------------------------------------------");
    }

    protected void changeName(Item item) throws Exception {
        String name = promptName(item);
        item.setName(name);
        try {
            Item updatedItem = auctionService.storeItem(item);
            // Stary przedmiot jest teraz nieaktualny, ale to w istocie nie ma
            // znaczenia, ponieważ i tak chcemy zrestartować dialog
            // po zakończeniu działania tej metody!
            System.out.println("=> Pomyślnie zmieniono nazwę przedmiotu!");
        } catch (Exception ex) {
            Throwable cause = Exceptions.unwrap(ex);

            if (cause instanceof OptimisticLockException)
                System.err.println(
                    "=> Przykro mi! Przedmiot, z którym pracowałeś został zmodyfikowany przez innego użytkownika."
                );
            else if (cause instanceof ConstraintViolationException)
                System.err.println(
                    "=> Popraw błędy walidacji zgłoszone przez serwer: " +
                        ((ConstraintViolationException)cause).getConstraintViolations()
                );
            else
                throw ex;
        }
    }

    protected void placeBid(Item item) throws Exception {
        Bid bid = promptBid(item); // Przedmiot jest w stanie odłączonym. Brak leniwego ładowania!
        try {
            Item updatedItem = auctionService.placeBid(bid);
            // Stary przedmiot jest teraz nieaktualny, ale to w istocie nie ma
            // znaczenia, ponieważ i tak chcemy zrestartować dialog
            // po zakończeniu działania tej metody!
            System.out.println("=> Pomyślnie złożono ofertę!");
        } catch (InvalidBidException ex) {
            System.out.println("Niewłaściwa oferta: " + ex.getMessage());
            placeBid(item);
        } catch (Exception ex) {
            Throwable cause = Exceptions.unwrap(ex);
            if (cause instanceof OptimisticLockException)
                System.err.println(
                    "=> Przykro mi! Przedmiot został zmodyfikowany przez innego użytkownika, albo złożono nową ofertę."
                );
            else
                throw ex;
        }
    }

    protected Item promptItem(List<Item> items) throws Exception {
        System.out.println("Wprowadź identyfikator przedmiotu:");
        try {
            Long id = Long.valueOf(userInput.readLine());
            for (Item item : items) {
                if (item.getId().equals(id))
                    return item;
            }
        } catch (NumberFormatException ex) {
            // Nie można odczytać danych wejściowych
            promptItem(items);
            return null;
        }
        System.out.println("Nie znaleziono przedmiotu!");
        promptItem(items);
        return null;
    }

    protected String promptName(Item item) throws Exception {
        System.out.println("Nowa nazwa przedmiotu '" + item.getName() + "':");
        // W rzeczywistym kliencie tutaj byłoby przekształcenie i sprawdzenie poprawności danych wejściowych...
        return userInput.readLine();
    }

    protected Bid promptBid(Item item) throws Exception {
        System.out.println("Twoja oferta na przedmiot '" + item.getName() + "':");
        BigDecimal amount = new BigDecimal(userInput.readLine());
        return new Bid(amount, item);
    }

    protected static <T> T lookupService(Class<T> remoteInterface, Class<? extends T> bean) throws NamingException {
        // To boli... ale przynajmniej część jest ustandaryzowana...
        final Hashtable jndiProperties = new Hashtable();
        jndiProperties.put(Context.URL_PKG_PREFIXES, "org.jboss.ejb.client.naming");
        Context context = new InitialContext(jndiProperties);

        String appName = ""; // Puste w przypadku instalacji WAR
        String moduleName = "app-stateless-server"; // Nazwa WAR (ma sens, prawda?)
        String distinctName = ""; // Jeśli nie wykonujemy więcej konfiguracyjnej „magii”, ta wartość jest zawsze pusta
        String beanName = bean.getSimpleName(); // To jest implementacja...
        String viewClassName = remoteInterface.getName(); // ... tego interfejsu
        String ejbName = "ejb:" + appName + "/" + moduleName + "/" + distinctName + "/" + beanName + "!" + viewClassName;
        return (T) context.lookup(ejbName);
    }
}

