package org.jpwh.test.stateful;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.persistence.UsingDataSet;
import org.jboss.arquillian.testng.Arquillian;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.jboss.shrinkwrap.resolver.api.maven.Maven;
import org.jpwh.model.InvalidBidException;
import org.jpwh.model.ItemBidSummary;
import org.jpwh.shared.util.Exceptions;
import org.jpwh.stateful.AuctionService;
import org.jpwh.stateful.ItemService;
import org.testng.Assert;
import org.testng.annotations.Test;

import javax.ejb.EJBException;
import javax.enterprise.inject.Instance;
import javax.inject.Inject;
import java.math.BigDecimal;
import java.util.List;

import static org.testng.Assert.*;

@UsingDataSet("testdata.xml")
public class AuctionServiceTest extends Arquillian {

    @Deployment
    public static WebArchive createTestArchive() {
        WebArchive archive = ShrinkWrap.create(WebArchive.class)
            .addPackages(true, "org.jpwh.stateful") // Rekurencyjne dodanie wszystkich klas w tym pakiecie
            .addAsLibraries(Maven.resolver() // Umieść zależności Maven w WEB-INF/lib
                .loadPomFromFile("pom.xml")
                .importRuntimeDependencies()
                .resolve()
                .withTransitivity()
                .asFile());

        System.out.println("=> Instalacja wirtualnego pakietu WAR testów integracyjnych:");
        System.out.println(archive.toString(true));
        return archive;
    }

    @Inject
    AuctionService auctionService;

    @Inject
    Instance<ItemService> itemServiceInstance;

    @Test
    public void editItemName() throws Exception {

        // Klient nie musi utrzymywać stanu. Renderuje listę obiektów DTO
        List<ItemBidSummary> itemBidSummaries = auctionService.getSummaries();

        // Konieczne są tylko wartości identyfikatora. Pobranie pierwszego na liście
        Long itemId = itemBidSummaries.get(0).getItemId();

        // Zmiana nazwy w konwersacji z egzemplarzem usługi stateful
        ItemService itemService = itemServiceInstance.get();
        itemService.startConversation(itemId);
        itemService.setItemName("Piękna rękawica do baseballa");

        // Nadal stara nazwa do czasu zatwierdzenia i zsynchroniowania rozszerzonego kontekstu utrwalania!
        itemBidSummaries = auctionService.getSummaries();
        assertEquals(itemBidSummaries.get(0).getName(), "Rękawica do Baseballa");

        // Rozpoczęcie współbieżnej, drugiej konwersacji
        ItemService concurrentItemService = itemServiceInstance.get();
        concurrentItemService.startConversation(itemId);
        concurrentItemService.setItemName("Brzydka rękawica do Baseballa");

        // Zatwierdzenie pierwszej konwersacji
        itemService.commitConversation();

        // Druga konwersacja nie może się powieść
        boolean test = false;
        try {
            concurrentItemService.commitConversation();
        } catch (EJBException ex) {
            // TODO: To powinien być wyjątek OptimisticLockException
            if (Exceptions.unwrap(ex) instanceof org.hibernate.StaleObjectStateException)
                test = true;
        }
        assertTrue(test, "należało zgłosić wyjątek StaleObjectStateException");

        // Pierwsza konwersacja zwycięża, ustawiono nową nazwę
        itemBidSummaries = auctionService.getSummaries();
        assertEquals(itemBidSummaries.get(0).getName(), "Piękna rękawica do baseballa");
    }

    @Test
    public void placeBid() throws Exception {
        // Klient nie musi utrzymywać stanu. Renderuje listę obiektów DTO
        List<ItemBidSummary> itemBidSummaries = auctionService.getSummaries();

        // Pobranie pierwszego obiektu DTO
        ItemBidSummary itemBidSummary = itemBidSummaries.get(0);

        // Sprawdzenie bieżącej liczby ofert i zgłoszenie nowej, wyższej oferty
        BigDecimal newBidAmount = new BigDecimal(itemBidSummary.getHighestBid().intValue() + 1);

        // Złożenie oferty w konwersacji z egzemplarzem usługi stateful
        ItemService itemService = itemServiceInstance.get();
        itemService.startConversation(itemBidSummary.getItemId());
        itemService.placeBid(newBidAmount);

        // Nadal stara oferta do czasu zatwierdzenia i zsynchroniowania rozszerzonego kontekstu utrwalania!
        itemBidSummaries = auctionService.getSummaries();
        assertEquals(itemBidSummaries.get(0).getHighestBid().compareTo(newBidAmount), -1);

        // Rozpoczęcie współbieżnej, drugiej konwersacji
        ItemService concurrentItemService = itemServiceInstance.get();
        concurrentItemService.startConversation(itemBidSummary.getItemId());
        BigDecimal secondNewBid = new BigDecimal(itemBidSummary.getHighestBid().intValue() + 2);
        concurrentItemService.placeBid(secondNewBid);

        // Zatwierdzenie pierwszej konwersacji
        itemService.commitConversation();

        // Druga konwersacja nie może się powieść
        boolean test = false;
        try {
            concurrentItemService.commitConversation();
        } catch (EJBException ex) {
            // TODO: To powinien być wyjątek OptimisticLockException
            if (Exceptions.unwrap(ex) instanceof org.hibernate.StaleObjectStateException)
                test = true;
        }
        assertTrue(test, "Powinien być zgłoszony wyjątek StaleObjectStateException");

        BigDecimal lowBidAmount = new BigDecimal(itemBidSummary.getHighestBid().intValue()-1);
        itemService = itemServiceInstance.get();
        itemService.startConversation(itemBidSummary.getItemId());
        test = false;
        try {
            itemService.placeBid(lowBidAmount);
        } catch (InvalidBidException ex) {
            // Zaległa konwersacja nigdy nie zostanie zatwierdzona. Następuje przekroczenie czasu działania komponentu bean stateful
            test = true;
        }
        Assert.assertTrue(test, "Powinien być zgłoszony wyjątek InvalidBidException");

        // Pierwsza konwersacja zwycięża, złożono nową ofertę
        itemBidSummaries = auctionService.getSummaries();
        assertEquals(itemBidSummaries.get(0).getHighestBid().compareTo(newBidAmount), 0);
    }
}

