package org.jpwh.stateful;

import org.hibernate.StaleObjectStateException;
import org.jpwh.model.InvalidBidException;
import org.jpwh.model.ItemBidSummary;
import org.jpwh.shared.util.Exceptions;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.persistence.OptimisticLockException;
import javax.validation.ConstraintViolationException;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.logging.LogManager;

public class AuctionClient {

    public static void main(String[] args) throws Exception {
        // Jeśli zostanie uruchomiona ta metoda main() za pomocą wtyczki Maven exec, to JVM jest już uruchomiona i nie będzie
        // rozwidlona. Teraz jest już za późno, aby mechanizm JUL automatycznie pobrał tę właściwość systemu. Albo coś innego...
        LogManager.getLogManager().readConfiguration(
            new FileInputStream(new File(System.getProperty("java.util.logging.config.file"))));

        AuctionClient client = new AuctionClient();
        client.startDialog();
    }

    protected final DateFormat dateFormat = new SimpleDateFormat("dd. MMM yyyy HH:mm", Locale.ENGLISH);
    protected final BufferedReader userInput = new BufferedReader(new InputStreamReader(System.in));

    protected final RemoteAuctionService auctionService;

    public AuctionClient() throws Exception {
        System.out.println(">>> Rozpoczęcie dialogu, nawiązanie połączenia z serwerem (wciśnij CTRL+C, aby zakończyć)...");
        this.auctionService = lookupService(RemoteAuctionService.class, AuctionServiceImpl.class, false);
    }

    @SuppressWarnings("InfiniteRecursion")
    public void startDialog() throws Exception {

         // Pobranie listy obiektów DTO z usługi. Bezstanowo
        List<ItemBidSummary> itemBidSummaries = auctionService.getSummaries();

        // Wyrenderowanie listy
        renderItemBidSummary(itemBidSummaries);

         // Umożliwienie użytkownikowi wskazania identyfikatora przedmiotu i akcji do wykonania
        ItemBidSummary itemBidSummary = promptItemBidSummary(itemBidSummaries);
        System.out.println("Czy chcesz zmienić nazwę (n) przedmiot, czy złożyć ofertę (b):");
        String action = userInput.readLine();

        if (action.equals("n")) {

            RemoteItemService itemService = startConversation(itemBidSummary.getItemId());
            itemService.setItemName(promptName());
            commitConversation(itemService);

        } else if (action.equals("b")) {

            RemoteItemService itemService = startConversation(itemBidSummary.getItemId());
            // Ta operacja może zgłaszać odwracalne wyjątki. Dlatego została shermetyzowana
            placeBid(itemService, itemBidSummary);
            commitConversation(itemService);
        }

        startDialog();
    }

  protected void renderItemBidSummary(List<ItemBidSummary> itemBidSummaries) {
        if (itemBidSummaries.size() == 0) {
            System.err.println("Nie znaleziono przedmiotów. Czy na pewno zaimportowałeś dane testowe? Kończę pracę...");
            System.exit(1);
        }
        System.out.println("--------------------------------------------------------------------");
        String format = "%-4s | %-20s | %-20s | %15s %n";
        System.out.printf(format, "ID", "Nazwa", "Koniec aukcji", "Najwyższa oferta");
        System.out.println("--------------------------------------------------------------------");
        for (ItemBidSummary itemBidSummary : itemBidSummaries) {
            String highestBidAmount = itemBidSummary.getHighestBid() != null
                ? itemBidSummary.getHighestBid().toString()
                : "-";
            String auctionEnd = dateFormat.format(itemBidSummary.getAuctionEnd());
            System.out.printf(format, itemBidSummary.getItemId(), itemBidSummary.getName(), auctionEnd, highestBidAmount);
        }
        System.out.println("--------------------------------------------------------------------");
    }


    protected ItemBidSummary promptItemBidSummary(List<ItemBidSummary> itemBidSummaries) throws Exception {
        System.out.println("Wprowadź ID przedmiotu:");
        try {
            Long id = Long.valueOf(userInput.readLine());
            for (ItemBidSummary itemBidSummary : itemBidSummaries) {
                if (itemBidSummary.getItemId().equals(id))
                    return itemBidSummary;
            }
        } catch (NumberFormatException ex) {
            // Nie można odczytać danych wejściowych 
            promptItemBidSummary(itemBidSummaries);
            return null;
        }
        System.out.println("Nie znaleziono przedmiotu!");
        promptItemBidSummary(itemBidSummaries);
        return null;
    }

 protected String promptName() throws Exception {
        System.out.println("Wprowadź nową nazwę:");
        return userInput.readLine();
    }

    protected BigDecimal promptBid(ItemBidSummary itemBidSummary) throws Exception {
        System.out.println("Your bid for item '" + itemBidSummary.getName() + "':");
        return new BigDecimal(userInput.readLine());
    }

    protected void placeBid(RemoteItemService itemService, ItemBidSummary itemBidSummary) throws Exception {
        // Obsłużenie wyjątków, które nie „zabijają” konwersacji
        try {
            itemService.placeBid(promptBid(itemBidSummary));
        } catch (NumberFormatException ex ) { // Zgłoszone przez klienta!
            System.out.println("=> Przykro mi! To nie jest liczba. Spróbuj ponownie.");
            placeBid(itemService, itemBidSummary);
        } catch (InvalidBidException ex) { // Zgłoszony przez serwer!
            System.out.println("=> Przykro mi! Invalid bid, try again: " + ex.getMessage());
            placeBid(itemService, itemBidSummary);
        }
    }

    protected RemoteItemService startConversation(Long itemId) throws Exception {
        RemoteItemService itemService = lookupService(RemoteItemService.class, ItemServiceImpl.class, true);
        // ALBO ZA POMOCĄ DAO:
        // RemoteItemService itemService = lookupService(RemoteItemService.class, ItemServiceWithDAOImpl.class, true);

        System.out.println("=> Rozpoczęcie konwersacji z serwerem");
        itemService.startConversation(itemId);

        return itemService;
    }

    protected void commitConversation(RemoteItemService itemService) throws Exception {
        try {
            itemService.commitConversation();
            System.out.println("=> Konwersacja zatwierdzona pomyślnie!");
        } catch (Exception ex) {
            Throwable cause = Exceptions.unwrap(ex);

            // Wyjątki podczas zatwierdzania „zabijają” konwersację, ponieważ w przypadku, kiedy w metodzie @Remove commitConversation() zostanie sgłoszony wyjątek, 
            // na serwerze nie jest utrzymywany komponent bean sesji z obsługą stanów.
            // Wiemy, że w przypadku niektórych wyjątków możne wyświetlić opisowy komunikat o błędzie i zrestartować
            // konwersację. Nie trzeba zabijać całego klienta.
            if (cause instanceof OptimisticLockException || cause instanceof StaleObjectStateException)
                System.err.println(
                    "=> Przykro mi! Dane konwersacji zostały równolegle zmodyfikowane przez innego użytkownika."
                );
            else if (cause instanceof ConstraintViolationException)
                System.err.println(
                    "=> Przykro mi! Serwer zgłosił błędy walidacji: " +
                    ((ConstraintViolationException)cause).getConstraintViolations()
                );
            else
                throw ex;
        }
    }

    protected static <T> T lookupService(Class<T> remoteInterface, Class<? extends T> bean, boolean stateful) throws NamingException {
        // To boli... ale przynajmniej część jest ustandaryzowana...
        final Hashtable jndiProperties = new Hashtable();
        jndiProperties.put(Context.URL_PKG_PREFIXES, "org.jboss.ejb.client.naming");
        Context context = new InitialContext(jndiProperties);

        String appName = ""; // Puste w przypadku instalacji WAR
        String moduleName = "app-stateful-server"; // Nazwa WAR (ma sens, prawda?)
        String distinctName = ""; // Jeśli nie wykonujemy więcej konfiguracyjnej „magii”, ta wartość jest zawsze pusta
        String beanName = bean.getSimpleName(); // To jest implementacja...
        String viewClassName = remoteInterface.getName(); // ... tego interfejsu
        String ejbName = "ejb:" + appName + "/" + moduleName + "/" + distinctName + "/" + beanName + "!" + viewClassName;
        if (stateful)
            ejbName = ejbName + "?stateful";
        return (T) context.lookup(ejbName);
    }
}

