package org.jpwh.test.concurrency;

import org.jpwh.env.JPATest;
import org.jpwh.model.concurrency.version.Bid;
import org.jpwh.model.concurrency.version.Category;
import org.jpwh.model.concurrency.version.InvalidBidException;
import org.jpwh.model.concurrency.version.Item;
import org.jpwh.shared.util.TestData;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.persistence.LockModeType;
import javax.persistence.NoResultException;
import javax.persistence.OptimisticLockException;
import javax.transaction.UserTransaction;
import java.math.BigDecimal;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;

import static org.testng.Assert.assertEquals;

public class Versioning extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("ConcurrencyVersioningPU");
    }

    @Test(expectedExceptions = OptimisticLockException.class)
    public void firstCommitWins() throws Throwable {
        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();
            Item someItem = new Item("Jakiś przedmiot");
            em.persist(someItem);
            tx.commit();
            em.close();
            final Long ITEM_ID = someItem.getId();

            tx.begin();
            em = JPA.createEntityManager();

            /* 
               Pobranie egzemplarza encji według identyfikatora powoduje załadowanie bieżącej wersji z bazy danych za pomocą instrukji SELECT.
             */
            Item item = em.find(Item.class, ITEM_ID);
            // select * from ITEM where ID = ?

            /* 
               Bieżąca wersja egzemplarza  <code>Item</code> to 0.
             */
            assertEquals(item.getVersion(), 0);

            item.setName("Nowa nazwa");

            // Współbieżna, druga jednostka pracy wykonuje to samo
            Executors.newSingleThreadExecutor().submit(new Callable<Object>() {
                @Override
                public Object call() throws Exception {
                    UserTransaction tx = TM.getUserTransaction();
                    try {
                        tx.begin();
                        EntityManager em = JPA.createEntityManager();

                        Item item = em.find(Item.class, ITEM_ID);
                        // select * from ITEM where ID = ?

                        assertEquals(item.getVersion(), 0);

                        item.setName("Inna nazwa");

                        tx.commit();
                        // update ITEM set NAME = ?, VERSION = 1 where ID = ? and VERSION = 0
                        // Ta instrukcja kończy się sukcesem. W bazie danych istnieje wiersz z ID = ? i VERSION = 0!
                        em.close();

                    } catch (Exception ex) {
                        // To nie powinno się zdarzyć. Ta operacja zatwierdzania powinna zwyciężyć!
                        TM.rollback();
                        throw new RuntimeException("Błąd operacji współbieżnej: " + ex, ex);
                    }
                    return null;
                }
            }).get();

            /* 
               Podczas synchronizacji kontekstu utrwalania Hibernate wykrywa zmodyfikowany egzemplarz Item i inkrementuje jego wersję do 1. 
               Instrukcja SQL UPDATE wykonuje teraz sprawdzenie wersji, i zapisuje nową wersję w bazie danych, ale tylko wtedy, gdy wersja bazy danych nadal wynosi 0.
             */
            em.flush();
            // update ITEM set NAME = ?, VERSION = 1 where ID = ? and VERSION = 0

        } catch (Exception ex) {
            throw unwrapCauseOfType(ex, OptimisticLockException.class);
        } finally {
            TM.rollback();
        }
    }

    // TODO Ten kod zgładza nieprawidłowy wyjątek!
    // @Test(expectedExceptions = OptimisticLockException.class)
    @Test(expectedExceptions = org.hibernate.OptimisticLockException.class)
    public void manualVersionChecking() throws Throwable {
        final ConcurrencyTestData testData = storeCategoriesAndItems();
        Long[] CATEGORIES = testData.categories.identifiers;

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            BigDecimal totalPrice = new BigDecimal(0);
            for (Long categoryId : CATEGORIES) {

                /* 
                   Dla każdej <code>Kategorii</code> odpytaj o wszystkie egzemplarze <code>Item</code> w trybie
                   blokady <code>OPTIMISTIC</code>. Hibernate będzie teraz wiedział, że musi sprawdzić każdy 
                   egzemplarz <code>Item</code> w czasie synchronizowania.
                 */
                List<Item> items =
                    em.createQuery("select i from Item i where i.category.id = :catId")
                        .setLockMode(LockModeType.OPTIMISTIC)
                        .setParameter("catId", categoryId)
                        .getResultList();

                for (Item item : items)
                    totalPrice = totalPrice.add(item.getBuyNowPrice());

                // Teraz współbieżna transakcja przeniesie przedmiot do innej kategorii
                if (categoryId.equals(testData.categories.getFirstId())) {
                    Executors.newSingleThreadExecutor().submit(new Callable<Object>() {
                        @Override
                        public Object call() throws Exception {
                            UserTransaction tx = TM.getUserTransaction();
                            try {
                                tx.begin();
                                EntityManager em = JPA.createEntityManager();

                                // Przeniesienie pierwszego elementu z pierwszej kategorii do ostatniej kategorii
                                List<Item> items =
                                    em.createQuery("select i from Item i where i.category.id = :catId")
                                        .setParameter("catId", testData.categories.getFirstId())
                                        .getResultList();

                                Category lastCategory = em.getReference(
                                    Category.class, testData.categories.getLastId()
                                );

                                items.iterator().next().setCategory(lastCategory);

                                tx.commit();
                                em.close();
                            } catch (Exception ex) {
                                // To nie powinno się zdarzyć. Ta operacja zatwierdzania powinna zwyciężyć!
                                TM.rollback();
                                throw new RuntimeException("Błąd operacji współbieżnej: " + ex, ex);
                            }
                            return null;
                        }
                    }).get();
                }
            }

            /* 
               Dla każdego egzemplarza <code>Item</code>  załadowanego wcześniej Hibernate wykonuje podczas synchronizowania instrukcję <code>SELECT</code>.
               Sprawdza, czy wersja bazy danych każdego wiersza tabeli <code>ITEM</code> jest nadal taka sama, jaka była podczas ładowania. 
               Jeżeli dowolny wiersz tabeli <code>ITEM</code>  ma inną wersję, albo wiersz już nie istnieje, zgłaszany jest wyjątek <code>OptimisticLockException</code>.
             */
            tx.commit();
            em.close();

            assertEquals(totalPrice.toString(), "108.00");
        } catch (Exception ex) {
            throw unwrapCauseOfType(ex, org.hibernate.OptimisticLockException.class);
        } finally {
            TM.rollback();
        }
    }

    // TODO Ten kod zgładza nieprawidłowy wyjątek!
    // @Test(expectedExceptions = OptimisticLockException.class)
    @Test(expectedExceptions = org.hibernate.StaleObjectStateException.class)
    public void firstCommitWins() throws Throwable {
        final TestData testData = storeItemAndBids();
        Long ITEM_ID = testData.getFirstId();

        UserTransaction tx = TM.getUserTransaction();
        try {

            tx.begin();
            EntityManager em = JPA.createEntityManager();

            /* 
               Metoda <code>find()</code> pobiera <code>LockModeType</code>. Tryb
               <code>OPTIMISTIC_FORCE_INCREMENT</code> zleca frameworkowi Hibernate inkrementację 
               wersji pobranego egzemplarza Item po załadowaniu z bazy danych nawet wtedy, gdy jednostka pracy nie zmodyfikowała tego egzemplarza.
             */
            Item item = em.find(
                Item.class,
                ITEM_ID,
                LockModeType.OPTIMISTIC_FORCE_INCREMENT
            );

            Bid highestBid = queryHighestBid(em, item);

            // Teraz współbieżna transakcja złoży ofertę na ten przedmiot.
            // Operacja zakończy się sukcesem, ponieważ pierwsze zatwierdzenie wygrywa!
            Executors.newSingleThreadExecutor().submit(new Callable<Object>() {
                @Override
                public Object call() throws Exception {
                    UserTransaction tx = TM.getUserTransaction();
                    try {
                        tx.begin();
                        EntityManager em = JPA.createEntityManager();

                        Item item = em.find(
                            Item.class,
                            testData.getFirstId(),
                            LockModeType.OPTIMISTIC_FORCE_INCREMENT
                        );
                        Bid highestBid = queryHighestBid(em, item);
                        try {
                            Bid newBid = new Bid(
                                new BigDecimal("44.44"),
                                item,
                                highestBid
                            );
                            em.persist(newBid);
                        } catch (InvalidBidException ex) {
                            // Ignorujemy
                        }

                        tx.commit();
                        em.close();
                    } catch (Exception ex) {
                        // To nie powinno się zdarzyć. Ta operacja zatwierdzania powinna zwyciężyć!
                        TM.rollback();
                        throw new RuntimeException("Błąd operacji współbieżnej: " + ex, ex);
                    }
                    return null;
                }
            }).get();

            try {
                /* 
                   Kod utrwala nowy egzemplarz klasy <code>Bid</code>, nie ma to wpływu na wartości egzemplarza <code>Item</code>.
                   Do tabeli <code>BID</code> zostanie wstawiony nowy wiersz. Hibernate nie wykryłby równolegle składanych ofert bez 
                   wymuszonej inkrementacji wersji egzemplarza klasy <code>Item</code>. Użyliśmy także kontrolowanego wyjątku 
                   w celu weryfikacji kwoty nowej oferty. Kwota ta musi być większa niż aktualnie najwyższa oferta.
                */
                Bid newBid = new Bid(
                    new BigDecimal("44.44"),
                    item,
                    highestBid
                );
                em.persist(newBid);
            } catch (InvalidBidException ex) {
                // Oferta zbyt niska. Wyświetlamy okno z błędem weryfikacji poprawności...
            }

            /* 
               Podczas synchronizacji kontekstu utrwalania Hibernate uruchamia instrukcję INSERT dla nowego egzemplarza klasy Bid i 
               wymusza aktualizację egzemplarza Item ze sprawdzeniem wersji.
               Jeżeli ktoś równolegle zmodyfikował egzemplarz Item, albo złożył nową ofertę równolegle z tą procedurą, to Hibernate zgłosi wyjątek.
             */
            tx.commit();
            em.close();
        } catch (Exception ex) {
            throw unwrapCauseOfType(ex, org.hibernate.StaleObjectStateException.class);
        } finally {
            TM.rollback();
        }
    }

    /* ################################################################################### */

    class ConcurrencyTestData {
        TestData categories;
        TestData items;
    }

    public ConcurrencyTestData storeCategoriesAndItems() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();
        ConcurrencyTestData testData = new ConcurrencyTestData();
        testData.categories = new TestData(new Long[3]);
        testData.items= new TestData(new Long[5]);
        for (int i = 1; i <= testData.categories.identifiers.length; i++) {
            Category category = new Category();
            category.setName("Category: " + i);
            em.persist(category);
            testData.categories.identifiers[i - 1] = category.getId();
            for (int j = 1; j <= testData.categories.identifiers.length; j++) {
                Item item = new Item("Przedmiot " + j);
                item.setCategory(category);
                item.setBuyNowPrice(new BigDecimal(10 + j));
                em.persist(item);
                testData.items.identifiers[(i - 1) + (j - 1)] = item.getId();
            }
        }
        tx.commit();
        em.close();
        return testData;
    }

    public TestData storeItemAndBids() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();
        Long[] ids = new Long[1];
        Item item = new Item("Jakiś przedmiot");
        em.persist(item);
        ids[0] = item.getId();
        for (int i = 1; i <= 3; i++) {
            Bid bid = new Bid(new BigDecimal(10 + i), item);
            em.persist(bid);
        }
        tx.commit();
        em.close();
        return new TestData(ids);
    }

    protected Bid queryHighestBid(EntityManager em, Item item) {
        // Nie można przewijać kursorów w JPA. Trzeba użyć setMaxResult()
        try {
            return (Bid) em.createQuery(
                "select b from Bid b" +
                    " where b.item = :itm" +
                    " order by b.amount desc"
            )
                .setParameter("itm", item)
                .setMaxResults(1)
                .getSingleResult();
        } catch (NoResultException ex) {
            return null;
        }
    }

}
